/*
 * Decompiled with CFR 0.152.
 */
package org.apache.tinkerpop.gremlin.driver;

import io.netty.bootstrap.Bootstrap;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelHandler;
import io.netty.channel.ChannelPromise;
import io.netty.channel.socket.nio.NioSocketChannel;
import io.netty.util.concurrent.GenericFutureListener;
import java.net.URI;
import java.time.Instant;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.tinkerpop.gremlin.driver.Channelizer;
import org.apache.tinkerpop.gremlin.driver.Client;
import org.apache.tinkerpop.gremlin.driver.Cluster;
import org.apache.tinkerpop.gremlin.driver.ConnectionPool;
import org.apache.tinkerpop.gremlin.driver.Result;
import org.apache.tinkerpop.gremlin.driver.ResultQueue;
import org.apache.tinkerpop.gremlin.driver.ResultSet;
import org.apache.tinkerpop.gremlin.driver.exception.ConnectionException;
import org.apache.tinkerpop.gremlin.driver.handler.IdleConnectionHandler;
import org.apache.tinkerpop.gremlin.util.message.RequestMessage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class Connection {
    public static final int MAX_WAIT_FOR_CONNECTION = 16000;
    public static final int MAX_WAIT_FOR_CLOSE = 3000;
    public static final long MAX_RESPONSE_CONTENT_LENGTH = Integer.MAX_VALUE;
    public static final int RECONNECT_INTERVAL = 1000;
    public static final int RESULT_ITERATION_BATCH_SIZE = 64;
    public static final long CONNECTION_SETUP_TIMEOUT_MILLIS = 15000L;
    public static final long CONNECTION_IDLE_TIMEOUT_MILLIS = 180000L;
    private static final Logger logger = LoggerFactory.getLogger(Connection.class);
    private final Channel channel;
    private final URI uri;
    private final AtomicReference<ResultQueue> pending = new AtomicReference();
    private final Cluster cluster;
    private final Client client;
    private final ConnectionPool pool;
    private final String creatingThread;
    private final String createdTimestamp;
    private final AtomicBoolean isBorrowed = new AtomicBoolean(false);
    public final AtomicBoolean isBeingReplaced = new AtomicBoolean(false);
    private final String connectionLabel;
    private final Channelizer channelizer;
    private final AtomicReference<CompletableFuture<Void>> closeFuture = new AtomicReference();
    private final AtomicBoolean shutdownInitiated = new AtomicBoolean(false);

    public Connection(URI uri, ConnectionPool pool) throws ConnectionException {
        this.uri = uri;
        this.cluster = pool.getCluster();
        this.client = pool.getClient();
        this.pool = pool;
        this.creatingThread = Thread.currentThread().getName();
        this.createdTimestamp = Instant.now().toString();
        this.connectionLabel = "Connection{host=" + pool.host + "}";
        if (this.cluster.isClosing()) {
            throw new IllegalStateException("Cannot open a connection with the cluster after close() is called");
        }
        if (this.client.isClosing()) {
            throw new IllegalStateException("Cannot open a connection with the client after close() is called");
        }
        Bootstrap b = this.cluster.getFactory().createBootstrap();
        try {
            this.channelizer = new Channelizer.HttpChannelizer();
            this.channelizer.init(this);
            ((Bootstrap)b.channel(NioSocketChannel.class)).handler((ChannelHandler)this.channelizer);
            this.channel = b.connect(uri.getHost(), uri.getPort()).sync().channel();
            this.channelizer.connected();
            Connection thisConnection = this;
            this.channel.closeFuture().addListener((GenericFutureListener)((ChannelFutureListener)future -> {
                logger.debug("OnChannelClose callback called for channel {}", (Object)this.channel);
                if (thisConnection.closeFuture.get() == null) {
                    if (!this.channel.hasAttr(IdleConnectionHandler.IDLE_STATE_EVENT)) {
                        logger.error(String.format("Server closed the Connection on channel %s - scheduling removal from %s", this.channel.id().asShortText(), thisConnection.pool.getPoolInfo(thisConnection)));
                    }
                    thisConnection.cluster.connectionScheduler().submit(() -> thisConnection.pool.destroyConnection(thisConnection));
                }
            }));
            logger.debug("Created new connection for {}", (Object)uri);
        }
        catch (Exception ex) {
            throw new ConnectionException(uri, "Could not open " + this.getConnectionInfo(true), ex);
        }
    }

    public boolean isDead() {
        return this.channel != null && !this.channel.isActive();
    }

    public AtomicBoolean isBorrowed() {
        return this.isBorrowed;
    }

    boolean isClosing() {
        return this.closeFuture.get() != null;
    }

    URI getUri() {
        return this.uri;
    }

    Cluster getCluster() {
        return this.cluster;
    }

    AtomicReference<ResultQueue> getPending() {
        return this.pending;
    }

    public synchronized CompletableFuture<Void> closeAsync() {
        if (this.isClosing()) {
            return this.closeFuture.get();
        }
        CompletableFuture<Void> future = new CompletableFuture<Void>();
        this.closeFuture.set(future);
        if (this.isOkToClose()) {
            if (null == this.channel) {
                future.complete(null);
            } else {
                this.shutdown(future);
            }
        } else {
            new CheckForPending(future).runUntilDone(this.cluster.connectionScheduler());
        }
        return future;
    }

    public ChannelPromise write(RequestMessage requestMessage, CompletableFuture<ResultSet> resultQueueSetup) {
        Connection thisConnection = this;
        ChannelPromise requestPromise = this.channel.newPromise().addListener(f -> {
            if (!f.isSuccess()) {
                if (logger.isDebugEnabled()) {
                    logger.debug(String.format("Write on connection %s failed", thisConnection.getConnectionInfo()), f.cause());
                }
                this.handleConnectionCleanupOnError(thisConnection);
                this.cluster.executor().submit(() -> resultQueueSetup.completeExceptionally(f.cause()));
            } else {
                LinkedBlockingQueue<Result> resultLinkedBlockingQueue = new LinkedBlockingQueue<Result>();
                CompletableFuture<Void> readCompleted = new CompletableFuture<Void>();
                readCompleted.whenCompleteAsync((v, t) -> {
                    if (t != null) {
                        logger.debug("Error while processing request on the server {}.", (Object)this, t);
                        this.handleConnectionCleanupOnError(thisConnection);
                    } else {
                        thisConnection.returnToPool();
                    }
                    this.tryShutdown();
                }, (Executor)this.cluster.executor());
                ResultQueue handler = new ResultQueue(resultLinkedBlockingQueue, readCompleted);
                this.pending.set(handler);
                this.cluster.executor().submit(() -> resultQueueSetup.complete(new ResultSet(handler, this.cluster.executor(), readCompleted, requestMessage, this.pool.host)));
            }
        });
        this.channel.writeAndFlush((Object)requestMessage, requestPromise);
        return requestPromise;
    }

    private void returnToPool() {
        block3: {
            try {
                if (this.pool != null) {
                    this.pool.returnConnection(this);
                }
            }
            catch (ConnectionException ce) {
                if (!logger.isDebugEnabled()) break block3;
                logger.debug("Returned {} connection to {} but an error occurred - {}", new Object[]{this.getConnectionInfo(), this.pool, ce.getMessage()});
            }
        }
    }

    private void handleConnectionCleanupOnError(Connection thisConnection) {
        if (thisConnection.isDead()) {
            if (this.pool != null) {
                this.pool.replaceConnection(thisConnection);
            }
        } else {
            thisConnection.returnToPool();
        }
    }

    private boolean isOkToClose() {
        return this.pending.get() == null || this.channel != null && !this.channel.isOpen() || !this.pool.host.isAvailable();
    }

    private void tryShutdown() {
        if (this.isClosing() && this.isOkToClose()) {
            this.shutdown(this.closeFuture.get());
        }
    }

    private void shutdown(CompletableFuture<Void> future) {
        if (this.shutdownInitiated.compareAndSet(false, true)) {
            if (this.channelizer != null) {
                this.channelizer.close(this.channel);
            }
            if (this.channel != null) {
                ChannelPromise promise = this.channel.newPromise();
                promise.addListener(f -> {
                    if (f.cause() != null) {
                        future.completeExceptionally(f.cause());
                    } else {
                        if (logger.isDebugEnabled()) {
                            logger.debug("{} destroyed successfully.", (Object)this.getConnectionInfo());
                        }
                        future.complete(null);
                    }
                });
                if (!this.channel.closeFuture().isDone()) {
                    this.channel.close(promise);
                } else if (!promise.trySuccess()) {
                    logger.warn("Failed to mark a promise as success because it is done already: {}", (Object)promise);
                }
            } else {
                future.complete(null);
            }
        } else {
            future.complete(null);
        }
    }

    public String getConnectionInfo() {
        return this.getConnectionInfo(true);
    }

    public String getConnectionInfo(boolean showHost) {
        return showHost ? String.format("Connection{channel=%s host=%s isDead=%s borrowed=%s pending=%s markedReplaced=%s closing=%s created=%s thread=%s}", this.getChannelId(), this.pool.host.toString(), this.isDead(), this.isBorrowed().get(), this.getPending().get() == null ? 0 : 1, this.isBeingReplaced, this.isClosing(), this.createdTimestamp, this.creatingThread) : String.format("Connection{channel=%s isDead=%s borrowed=%s pending=%s markedReplaced=%s closing=%s created=%s thread=%s}", this.getChannelId(), this.isDead(), this.isBorrowed().get(), this.getPending().get() == null ? 0 : 1, this.isBeingReplaced, this.isClosing(), this.createdTimestamp, this.creatingThread);
    }

    String getChannelId() {
        return this.channel != null ? this.channel.id().asShortText() : "null";
    }

    public String toString() {
        return String.format(this.connectionLabel + ", {channel=%s}", this.getChannelId());
    }

    private final class CheckForPending
    implements Runnable {
        private volatile ScheduledFuture<?> self;
        private final CompletableFuture<Void> future;
        private long checkUntil = System.currentTimeMillis();

        CheckForPending(CompletableFuture<Void> future) {
            this.future = future;
            this.checkUntil += (long)Connection.this.cluster.getMaxWaitForClose();
        }

        @Override
        public void run() {
            logger.info("Checking for pending messages to complete before close on {}", (Object)this);
            if (Connection.this.isOkToClose() || System.currentTimeMillis() > this.checkUntil) {
                Connection.this.shutdown(this.future);
                boolean interrupted = false;
                try {
                    while (null == this.self) {
                        try {
                            Thread.sleep(1L);
                        }
                        catch (InterruptedException e) {
                            interrupted = true;
                        }
                    }
                    this.self.cancel(false);
                }
                finally {
                    if (interrupted) {
                        Thread.currentThread().interrupt();
                    }
                }
            }
        }

        void runUntilDone(ScheduledExecutorService executor) {
            this.self = executor.scheduleAtFixedRate(this, 1000L, 1000L, TimeUnit.MILLISECONDS);
        }
    }
}

