// T-EXEC.CPP
// ==========
//
//   Executes a program of choice at the end of a file transfer.
//   The program to execute depends on the type of transfer, and is
//   read from the T-EXEC.INI file. The .ini file supports a number
//   of parameter which get substituted. They are:
//
//              %f = full path/file name of file transferred
//              %d = duration of transfer in ms.
//              %c = sub-event code number
//              %u = user name that transferred the file
//              %i = IP address of FTP client
//              %h = server home IP address
//
//   Syntax for that file is (example):
//
//      ********** T-EXEC.INI ********
//
//              [EXEC]
//              UploadSuccess=c:\program files\vscan\scan.exe %f
//              DownloadSuccess=c:\command.com /c c:\batch\countdown.bat %f
//              UploadFailed=c:\logs\error.exe %f %h %i %u %c %d
//              DownloadFailed=c:\logs\error.exe %f %h %i %u %c %d
//
//      *****************************
//
//   Note that by executing the command interpreter (command.com in Win95/98,
//   or cmd.exe in NT) one can also invoke batch files.
//
//             Author: Rob Beckers
//               Date: 04-MAR-99
//      Last revision: 04-MAR-99
//        Revision nr: 1
//
//   Revision history:
//

#include <windows.h>
#include <stdio.h>
#include <string.h>
#pragma hdrstop                         // to make Borland compiler stop including pre-compiled header files
#include "t-exec.h"

// static vars
char ComUpSuccess[MAXPATH];             // upload success command line
char ComUpFail[MAXPATH];                // upload failure command line
char ComDownSuccess[MAXPATH];           // download success command line
char ComDownFail[MAXPATH];              // download failure command line
UINT ExecRet;

BOOL WINAPI DllEntryPoint(HINSTANCE hInstance,DWORD fdwReason,PVOID /*pvReserved*/)
// ******
// Window's DLL entry point.
// Used for initializing list of names-to-block.
// ******
{
  switch (fdwReason) {
    case DLL_PROCESS_ATTACH:
    case DLL_THREAD_ATTACH:
      LoadIniFile(hInstance);           // load .ini file info
      break;
  }

  return(TRUE);                         // signal success to Windows
}

WORD CALLBACK HandleEventHook(RFTPEventStr* pEventStruc)
// ******
// Event (hook) handler.
// ******
{
  char ComLine[2*MAXPATH];              // command line to execute

  // disect event
  switch (pEventStruc->Event) {

    // successful upload
    case EVNT_EndUp:
      if (ComUpSuccess[0]=='\0') return(REVNT_None);
      ParseComLine(ComLine,ComUpSuccess,pEventStruc);
      break;

    // successful download
    case EVNT_EndDown:
      if (ComDownSuccess[0]=='\0') return(REVNT_None);
      ParseComLine(ComLine,ComDownSuccess,pEventStruc);
      break;

    // failed upload
    case EVNT_AbortUp:
      if (ComUpFail[0]=='\0') return(REVNT_None);
      ParseComLine(ComLine,ComUpFail,pEventStruc);
      break;

    // failed download
    case EVNT_AbortDown:
      if (ComDownFail[0]=='\0') return(REVNT_None);
      ParseComLine(ComLine,ComDownFail,pEventStruc);
      break;

    default:
      return(REVNT_None);               // don't handle anything else
  }

  ExecRet=WinExec(ComLine,SW_SHOWMINIMIZED);    // try to execute command line
  return(REVNT_None);
}

void LoadIniFile(HINSTANCE hInstance)
// ******
// Load .ini file variables
// ******
{
  char IniFile[MAXPATH];                // full path of .ini file

  // only initialize once
  static bool Once=0;
  if (Once) return;                     // already initialized
  Once=1;                               // this is first time, remember

  // construct full path of .ini file
  GetModuleFileName(hInstance,IniFile,MAXPATH);
  char* pEnd=strrchr(IniFile,'\\');	// strip .dll file name
  if (pEnd) pEnd[1]='\0';
  strcat(IniFile,INIFILE);              // create full path .ini name

  // get settings
  GetPrivateProfileString(INISECTION,INIUPSUC,"",ComUpSuccess,MAXPATH,IniFile);
  GetPrivateProfileString(INISECTION,INIUPFAIL,"",ComUpFail,MAXPATH,IniFile);
  GetPrivateProfileString(INISECTION,INIDOWNSUC,"",ComDownSuccess,MAXPATH,IniFile);
  GetPrivateProfileString(INISECTION,INIDOWNFAIL,"",ComDownFail,MAXPATH,IniFile);
}

void ParseComLine(char* pComOut,char* pComIn,RFTPEventStr* pEventStruc)
// ******
// Parse a command line for a particular event. Turn it into a command
// that can be executed.
//
// Parameters:     pComOut = (output) resulting command line to execute
//                  pComIn = (input) command line template
//             pEventStruc = (input) event structure for current event
// ******
{
  // copy input command line to output and process meta chars
  while (*pComIn!='\0') {

    // scan and process meta chars
    if (*pComIn=='%') {
      switch (pComIn[1]) {
        case 'f':                       // file name
          strcpy(pComOut,pEventStruc->AuxOne);
          pComOut+=strlen(pComOut);
          pComIn+=2;
          break;
        case 'd':                       // duration
          sprintf(pComOut,"%u",pEventStruc->Duration);
          pComOut+=strlen(pComOut);
          pComIn+=2;
          break;
        case 'c':                       // sub event code
          sprintf(pComOut,"%u",pEventStruc->SubEvent);
          pComOut+=strlen(pComOut);
          pComIn+=2;
          break;
        case 'u':                       // user name
          strcpy(pComOut,pEventStruc->User);
          pComOut+=strlen(pComOut);
          pComIn+=2;
          break;
        case 'i':                       // client IP
          strcpy(pComOut,pEventStruc->ClientIP);
          pComOut+=strlen(pComOut);
          pComIn+=2;
          break;
        case 'h':                       // server (home) IP
          strcpy(pComOut,pEventStruc->LocalIP);
          pComOut+=strlen(pComOut);
          pComIn+=2;
          break;
        default:                        // just copy anything we don't know
          *pComOut='%';
          pComOut++;
          pComIn++;
      }
    }

    // copy regular chars
    else {
      *pComOut=*pComIn;
      pComOut++;
      pComIn++;
    }
  }

  // done, terminate output string
  *pComOut='\0';
}
