// LIST.H
// ======
//
//   Template class, providing simple linked list management
//   for arbitrary object types.
//
//          Author: Rob Beckers
//            Date: 21-JUL-93
//   Last revision: 19-MAR-96
//     Revision nr: 6
//
//   Revision history:
//     26-JUL-93 : Fully tested and declared working
//     28-JUL-93 : Added 'Find' and consistent equality checking
//      1-AUG-93 : Added 'FindNext'
//     21-AUG-93 : Added 'Flush'
//      7-SEP-93 : Added 'NrItems'
//     20-NOV-93 : Fixed bug in 'Remove', deleting second of a list of 2 items set 'Head' to NULL
//     10-APR-94 : Made 'CurPoint' consistant, added 'GetCurItemPoint()'
//	1-JAN-95 : Added indexing and improved performance speed
//     19-MAR-96 : Added support for keeping a 'pool' of items, to avoid creation/deletion overhead
//
//   Remarks:
//     To test for equality or non-equality, only the '==' operator
//     is used. The left operant is always the input operant, the right
//     one is taken from the list.
//     So, to make these routines work on lists of structures, classes
//     etc, only an overloaded '==' function is needed.
//

#ifndef LISTH // to avoid multiple inclusion
#define LISTH

// work-around for a BC 3.1 bug - causing wrong debug info in .obj's
// for inline functions. Define 'DEBUG' for debugging.
#ifdef DEBUG
  #ifndef INLINE
    #define INLINE        // this is the INLINE definition for the production version
  #endif
#else
  #ifndef INLINE
    #define INLINE inline // this is the INLINE definition for debugging
  #endif
#endif

#include <stdlib.h> // for NULL definition

template <class T>
class RListElem
// ******
// Defines elements for our SimpleList
// ******
{
  public:
    RListElem(const T& Item) :
      Object(Item) {Next=NULL;};// constructor: automatically set item
    T Object;			// stored object
    RListElem<T> *Next;		// pointer to next item in SimpleList
};

template <class T>
class RSimpleList
// ******
// Linked list manager. Handles linked lists of anything.
// ******
{
  public:
    RSimpleList();
    ~RSimpleList();
    void Add(const T& Item);	 		// add item to end of list
    void Remove(const T& Item);	 		// remove first occurance of item
    int GetIndex(T& Item,unsigned int Index); 	// get item at index
    void AddIndex(const T& Item,unsigned int Idx); // add item at index position
    void RemoveIndex(unsigned int Idx); 	// remove item at index position
    int GetFirst(T& Item);			// return first item in list
    int GetNext(T& Item);	 		// return next item
    T* GetCurItemPoint();			// return pointer to current item
    int Find(T& Item);				// search for first occurance of item
    int FindNext(T& Item);			// find next occurance of item (after Find)
    void Flush();				// deletes all elements of the list
    unsigned int NrItems();			// return number of items stored in list
    unsigned int CurIndexNr();			// return current index number
    void SetPooling(int DoPool);		// switches item pooling on/off

  protected:
    RListElem<T>* NewElem(const T& Item);       // get new list element
    void DelElem(RListElem<T>* pElem);          // discart list element

    RListElem<T> *CurPoint;			// pointer to current location in list
    RListElem<T> *Head;    			// pointer to head of list
    T FindItem;					// to store item we're looking for
    unsigned int TotalItems;			// total number of stored items
    unsigned int Index;				// current index in list (starting at 0)
    int Pool;					// TRUE if we want to keep discarted items in a pool
    RListElem<T> *PoolPoint;			// pointer to pool of discarted items
};

template <class T>
RSimpleList<T>::RSimpleList()
// ******
// Constructor.
// ******
{
  CurPoint=Head=PoolPoint=NULL;
  TotalItems=0;
  Pool=0;          			// default: keep no item pool
}

template <class T>
RSimpleList<T>::~RSimpleList()
// ******
// Destructor: delete all list records
// ******
{
  SetPooling(0);			// force pooling off - destroys pool
  Flush();                              // destroys list
  return;
}

template <class T>
void RSimpleList<T>::Add(const T& Item)
// ******
// Add item to list. Item is added at the end of the list.
// Current point and index in list are set to the new item.
// ******
{
  RListElem<T> *Point,*NewStr;

  // make new list element and fill it in
  NewStr=NewElem(Item);

  // check if list is empty - special case
  if (Head==NULL) {
    Head=NewStr;
    CurPoint=Head; Index=0;
    TotalItems=1;
    return;
  }

  // try to use current index to speed things up
  if (CurPoint) Point=CurPoint;
  else {
    Point=Head;
    Index=0;
  }

  // find end of list & add
  while (Point->Next!=NULL) {
    Point=Point->Next;
    Index++;
  }
  Point->Next=NewStr;

  // update Curpoint and Index
  CurPoint=NewStr;
  Index++;
  TotalItems++;
}

template <class T>
void RSimpleList<T>::AddIndex(const T& Item,unsigned int Idx)
// ******
// Add item at index number.
// If index is out of range, the item is added at the end of the list.
// Both current point and index are set to the inserted item.
//
// Parameters: Item = Reference to item to insert
// 	        Idx = index number to insert item
// ******
{
  RListElem<T> *NewStr;

  // check if index is in range, add to end otherwise
  if (Idx>=TotalItems) {
    Add(Item);				// regular 'Add()' can handle this
    return;
  }

  // make new list element and fill it in
  NewStr=NewElem(Item);

  // special case: insert at first place in list
  if (Idx==0) {
    NewStr->Next=Head;
    Head=NewStr;
    CurPoint=Head;
    Index=0;
    TotalItems++;
    return;
  }

  // find place to insert, try to use current index in list for speed
  if (!(CurPoint && Index<Idx)) {
    CurPoint=Head;
    Index=0;
  }
  while (Index<Idx-1) {
    CurPoint=CurPoint->Next;
    Index++;
  }

  // insert new item
  NewStr->Next=CurPoint->Next;
  CurPoint->Next=NewStr;
  CurPoint=NewStr;
  Index++;
  TotalItems++;
}

template <class T>
void RSimpleList<T>::RemoveIndex(unsigned int Idx)
// ******
// Remove item at index number.
// If index is out of range, no item is removed.
// The current point is set to the item immidiately after the removed one.
//
// Parameters: Idx = index number of item to remove
// ******
{
  RListElem<T> *Point;

  // check if index is in range
  if (Idx>=TotalItems) {
    return;
  }

  // special case: removing first item in list
  if (Idx==0) {
    Point=Head;
    Head=Head->Next;
    CurPoint=Head;
    Index=0;
    TotalItems--;
    DelElem(Point);
    return;
  }

  // find item before item to remove, try to use current index in list for speed
  if (!(CurPoint && Index<Idx)) {
    CurPoint=Head;
    Index=0;
  }
  while (Index<Idx-1) {
    CurPoint=CurPoint->Next;
    Index++;
  }

  // remove item
  Point=CurPoint->Next;
  CurPoint->Next=Point->Next;
  CurPoint=Point->Next;
  Index++;
  TotalItems--;
  DelElem(Point);
}

template <class T>
void RSimpleList<T>::Remove(const T& Item)
// ******
// Remove first occurance of Item from list.
// Current point is reset to beginning of list
// ******
{
  RListElem<T> *Point,*PrevPoint;

  // don't remove empty list
  if (Head==NULL) return;

  // find it
  Point=PrevPoint=Head;
  while (!(Item==Point->Object) && Point->Next) {
    PrevPoint=Point;
    Point=Point->Next;
  }

  // remove, but only if item exists!
  if (Item==Point->Object) {
    if (Point==Head) Head=Point->Next; // exeption for first element
      else PrevPoint->Next=Point->Next;
    DelElem(Point);
    TotalItems--;
  }

  // reset CurPoint and index
  CurPoint=Head; Index=0;
}

template <class T>
int RSimpleList<T>::GetIndex(T& Item,unsigned int IndexNr)
// ******
// Returns item at index number.
//
// Parameters: Item    = Set to retrieved item on return
//	       IndexNr = Index nr. to retrieve, starting at 0
//
// Returns: 1 = Index nr. found and returned
//	    0 = Index nr. out of range
// ******
{
  RListElem<T> *Point;

  // check sanity
  if (IndexNr>=TotalItems) return(0);

  // try to use current index to get there fast
  if (CurPoint && Index<=IndexNr) {
    Point=CurPoint;
  }
  else {
    Point=Head;
    Index=0;
  }

  // go to item
  while (Index<IndexNr) {
    Point=Point->Next;
    Index++;
  }

  // got it
  CurPoint=Point; 
  Item=Point->Object;
  return(1);
}

template <class T>
unsigned int RSimpleList<T>::CurIndexNr()
// ******
// Returns current index number.
// Only meaningfull after seeks, retrievals etc. (when index
// is set to a defined value. If no index is available (ie. not
// meaningfull) 0 is returned. Index starts at value 0 for first item.
// ******
{
  if (CurPoint) return(Index);
  else return(0);
}

template <class T>
int RSimpleList<T>::GetFirst(T& Item)
// ******
// Return first item in list. Resets pointer in list
// to first item.
//
// Parameter: Item = Reference to variable to put first item
//
// Returns: 1 = First value exists
//	    0 = Empty list
// ******
{
  // check for empty list
  if (Head==NULL) return(0);

  // OK, value available
  CurPoint=Head; Index=0;
  Item=Head->Object;
  return(1);
}

template <class T>
int RSimpleList<T>::GetNext(T& Item)
// ******
// Return next item in list.
//
// Parameter: Item = Reference to variable to put item
//
// Returns: 1 = Value is valid
//	    0 = Value invalid, end-of-list reached
// ******
{
  // check if CurPoint makes sense
  if (!CurPoint) return(0);

  // check for end-of-list
  if (CurPoint->Next==NULL) return(0);

  // OK, end not yet reached
  CurPoint=CurPoint->Next;
  Item=CurPoint->Object;
  Index++;
  return(1);
}

template <class T>
int RSimpleList<T>::Find(T& Item)
// ******
// Returns first occurance of an item.
// Item is set explicitely if found, to the value of the found item.
// This makes no difference for simple types like int's, but makes searching
// compound types like struct's and returning them possible. If Item is not
// found, it is not changed.
// The value of Item is stored and a pointer is kept to the found item, so the
// next occurance of Item can be found by using FindNext.
//
// Parameter: Item = reference to an item
//
// Returns: 1 = exists
//          0 = doesn't exist
// ******
{
  RListElem<T> *Point;

  // don't search in empty list
  if (!Head) return(0);

  // find it
  Point=Head; Index=0;
  while (!(Item==Point->Object) && Point->Next) {
    Point=Point->Next;
    Index++;
  }

  // check if we found it, and set item
  if (Item==Point->Object) {

    // set values for FindNext
    CurPoint=Point;
    FindItem=Item;

    // set return item
    Item=Point->Object;
    return(1);
  }

  // no, just found end-of-list
  CurPoint=NULL; Index=0;
  return(0);
}

template <class T>
int RSimpleList<T>::FindNext(T& Item)
// ******
// Find next occurance of FindItem set by Find.
//
// Parameter: Item = set to found item on return
//
// Returns: 1 = next occurance found
//	    0 = no next occurance
// ******
{
  // don't look beyond end-of-list
  if (!CurPoint || !CurPoint->Next) return(0);

  do {
    CurPoint=CurPoint->Next;
    Index++;
  }
  while (!(FindItem==CurPoint->Object) && CurPoint->Next);

  // return item if we found it, otherwise we're at end-of-list
  if (FindItem==CurPoint->Object) {
    Item=CurPoint->Object;
    return(1);
  }

  CurPoint=NULL; Index=0;
  return(0);
}

template <class T>
void RSimpleList<T>::Flush()
// ******
// Flush list - remove all items and leave an
// empty list.
// ******
{
  RListElem<T> *pCurPos,*pNextPos;

  // only if list is not empty
  if (!Head) return;

  // work down the list
  pCurPos=Head;
  do {
    pNextPos=pCurPos->Next;
    DelElem(pCurPos);
    pCurPos=pNextPos;
  } while (pCurPos);

  // some housekeeping
  CurPoint=Head=NULL;
  Index=0;
  TotalItems=0;
}

template <class T>
INLINE unsigned int RSimpleList<T>::NrItems()
// ******
// Returns total number of items stored in list.
// ******
{
  return(TotalItems);
}

template <class T>
INLINE T* RSimpleList<T>::GetCurItemPoint()
// ******
// Returns pointer to the current item so it can be directly
// changed. Only meaningfull if current item has been set via
// 'GetFirst()', 'GetNext()', 'FindFirst()' or 'FindNext()'.
//
// NOTE: A dirty trick to get direct access to the stored item.
//       Using this to change it is not recommended practise...
// ******
{
  // a very little bit of sanity checking
  if (!CurPoint) return(NULL);

  // let's hope 'CurPoint' is meaningfull, return pointer to item
  return(&(CurPoint->Object));
}

template <class T>
INLINE RListElem<T>* RSimpleList<T>::NewElem(const T& Item)
// ******
// Return a new list element, set to the item
// If pooling is 'on' and there are items in the pool, then
// these will be taken first.
// ******
{
  if (PoolPoint) {
    RListElem<T>* TmpPnt=PoolPoint;	// get first item from pool list
    PoolPoint=TmpPnt->Next;		// unlink from list
    TmpPnt->Next=NULL;			// fully recycle element
    TmpPnt->Object=Item;		// set value
    return(TmpPnt);
  }
  else {
    return(new RListElem<T>(Item));
  }
}

template <class T>
INLINE void RSimpleList<T>::DelElem(RListElem<T>* pElem)
// ******
// Delete list element.
// If pooling is switched on it is added to the pool.
// ******
{
  if (Pool) {
    pElem->Next=PoolPoint;		// link it into the pool list
    PoolPoint=pElem;			// set new start of pool list
  }
  else delete pElem;			// no pooling -> delete
}

template <class T>
void RSimpleList<T>::SetPooling(int DoPool)
// ******
// Switch pooling on/off.
// ******
{
  // switching pooling off while on previously?
  if (!DoPool && Pool) {
    RListElem<T> *pCurPos,*pNextPos;

    // only if pool is not empty
    if (PoolPoint) {

      // work down the list
      pCurPos=PoolPoint;
      do {
        pNextPos=pCurPos->Next;
        delete pCurPos;                 // delete pool items
        pCurPos=pNextPos;
      } while (pCurPos);
      PoolPoint=NULL;
    }
  }

  Pool=DoPool;				// set new value
}

#endif
