// FILTERDLL.CPP
// =============
//
//   Upload/rename filter DLL for use with Serv-U v2.2 and higher.
//   Prevents the uploading or renaming of certain file names/types.
//   Loads the file names, or wildcards (extensions) to block from a
//   file named FILTER.INI. To make management easier it is also possible
//   to specify exceptions to the filter rules.
//   Syntax for the FILTER.INI file is (example):
//
//      ********* FILTER.INI ********
//
//              [BLOCK]
//		Filter1=*.exe
//              Filter2=\serv-u\*
//              Filter3=name??.hq*
//
//              [ALLOW]
//              Path1=c:\users\ftp\upexe\*.exe
//              Path2=c:\serv-u\notes.txt
//
//      *****************************
//
//             Author: Rob Beckers
//               Date: 09-MAR-97
//      Last revision: 10-JAN-98
//        Revision nr: 2
//
//   Revision history:
//          10-JAN-98: Added support for exceptions to the filter rules.
//

#include <windows.h>
#include <stdio.h>
#include <cstring.h>
#include <string.h>
#pragma hdrstop                         // to make Borland compiler stop including pre-compiled header files
#include "list.h"                       // used here for simple list management
#include "filterdll.h"

// Global variable definitions
LSTSTRINGS FilterList;                  // list of filter strings, loaded from .ini file
LSTSTRINGS AllowList;                   // list of exceptions to the filter rules, from .ini file

char* strichr(const char* InStr,char Chr)
// ******
// Scan a string for the occurance of a given character.
// Equivalent to 'strchr()' but case insensitive.
//
// Parameters: InStr = string to scan
// 		 Chr = character to scan for
//
// Returns: Pointer to character in string, or NULL if not found.
// ******
{
  char* pString=const_cast<char*>(InStr);
  while (pString[0]!='\0') {
    if (toupper(pString[0])==toupper(Chr)) {
      return(pString);			// found it!
    }
    else pString++;			// to next character
  }

  // if we get here there was no match
  return(NULL);
}

bool MatchGlob(const char* pName,const char* pArg)
// ******
// Check if Name matches the argument.
// Argument can contain wildcards (?*).
//
// Parameters: pName = link name
//	        pArg = argument
//
// Returns: TRUE if match, FALSE if no match
// ******
{
  // first check shortcuts
  if (strcmp(pArg,"*.*")==0 || strcmp(pArg,"*")==0) return(true);

  // no luck with the shortcuts, long check
  char* pArgPoint=const_cast<char*>(pArg); // pointer into argument
  char* pLnkPoint=const_cast<char*>(pName);// pointer into path name
  while (true) {			   // loop until something breaks

    // break conditions: run out of strings
    if (pArgPoint[0]=='\0' && pLnkPoint[0]=='\0') return(true);
  					   // all matched - OK
    if (pArgPoint[0]=='\0') return(false); // run out of argument, but still name left - not OK
    if (pLnkPoint[0]=='\0') return(false); // run out of name, but still argument left - not OK

    char* pNext;
    switch(pArgPoint[0]) {

      case '*':                            // match wildcard '*'
        if (pArgPoint[1]=='\0') return(true);
        pNext=strichr(pLnkPoint,pArgPoint[1]);
        if (!pNext) return(false);
        pLnkPoint=pNext+1;		   // already matched the next char
        pArgPoint+=2;			   //    ,,      ,,    ,,  ,,    ,,
        break;

      case '?':				   // match wildcard '?'
        pLnkPoint++;			   // just advance
        pArgPoint++;
        break;

      default:                             // exact match
        if (toupper(pArgPoint[0])!=toupper(pLnkPoint[0])) return(false);
        pLnkPoint++;			   // just advance
        pArgPoint++;
        break;
    }
  }
}

void LoadFilterList(HINSTANCE hInstance)
// ******
// Load the list of files/extensions to block
// ******
{
  char Item[50]; char Text[MAXPATH];
  char IniFile[MAXPATH];                // full path of .ini file

  // clear old list items
  FilterList.Flush();

  // construct full path of .ini file
  GetModuleFileName(hInstance,IniFile,MAXPATH);
  char* pEnd=strrchr(IniFile,'\\');	// strip .dll file name
  if (pEnd) pEnd[1]='\0';
  strcat(IniFile,INIFILE);              // create full path .ini name

  // read in current list items
  int Num=1;
  while (TRUE) {
    sprintf(Item,"%s%d",INILINE,Num);   // prepare entry to look for
    int n=GetPrivateProfileString(INISECTION,Item,"",Text,MAXPATH,IniFile);
    if (!n) break;                      // done if no item found
    FilterList.Add(Text);               // add to filter list
    Num++;
  }
}

void LoadAllowList(HINSTANCE hInstance)
// ******
// Load the list of exceptions to the filter rules.
// ******
{
  char Item[50]; char Text[MAXPATH];
  char IniFile[MAXPATH];                // full path of .ini file

  // clear old list items
  AllowList.Flush();

  // construct full path of .ini file
  GetModuleFileName(hInstance,IniFile,MAXPATH);
  char* pEnd=strrchr(IniFile,'\\');	// strip .dll file name
  if (pEnd) pEnd[1]='\0';
  strcat(IniFile,INIFILE);              // create full path .ini name

  // read in current list items
  int Num=1;
  while (TRUE) {
    sprintf(Item,"%s%d",INIPATH,Num);   // prepare entry to look for
    int n=GetPrivateProfileString(INIALLOW,Item,"",Text,MAXPATH,IniFile);
    if (!n) break;                      // done if no item found
    AllowList.Add(Text);                // add to filter list
    Num++;
  }
}

bool CheckFilterFile(char* pPath)
// ******
// Check if a path/file is on the list of
// blocked files.
//
// Parameter: pPath = path of file to download
//
// Returns: TRUE if file should be blocked, FALSE if it should be allowed
// ******
{
  // check the list of rules
  char* pRule; char* pComp; string String;
  bool Block=false;                     // TRUE if we want to block
  if (FilterList.GetFirst(String)) do {

    // get a character pointer
    pRule=const_cast<char*>(String.c_str());

    // check if block rule has only file name
    if (!strchr(pRule,'\\')) {
      pComp=strrchr(pPath,'\\');	// get file name
      if (pComp) {
        pComp++;			// point to name only
        if (MatchGlob(pComp,pRule)) {
          Block=true;                   // file is on block list
          break;
        }
      }
    }

    // check if block rule misses drive letter
    else if (strlen(pRule)>2 && pRule[0]=='\\') {
      pComp=strchr(pPath,'\\');	        // ignore drive letter for comparison
      if (pComp && MatchGlob(pComp,pRule)) {
        Block=true;                     // file is on block list
        break;
      }
    }

    // only option left is that rule is complete path
    else if (MatchGlob(pPath,pRule)) {
      Block=true;                       // file is on block list
      break;
    }
  }
  while (FilterList.GetNext(String));

  // if file was on block list, check if it should be allowed according
  // to allow list
  if (!Block) return(false);            // not on block list - allow

  // file is on block list, check allow list for exceptions
  if (AllowList.GetFirst(String)) do {

    // get a character pointer
    pRule=const_cast<char*>(String.c_str());

    // check if block rule has only file name
    if (!strchr(pRule,'\\')) {
      pComp=strrchr(pPath,'\\');	// get file name
      if (pComp) {
        pComp++;			// point to name only
        if (MatchGlob(pComp,pRule)) {
          return(false);                // file is on allow list
        }
      }
    }

    // check if block rule misses drive letter
    else if (strlen(pRule)>2 && pRule[0]=='\\') {
      pComp=strchr(pPath,'\\');	        // ignore drive letter for comparison
      if (pComp && MatchGlob(pComp,pRule)) {
        return(false);                  // file is on allow list
      }
    }

    // only option left is that rule is complete path
    else if (MatchGlob(pPath,pRule)) {
      return(false);
    }
  }
  while (AllowList.GetNext(String));

  // we get here if file is on block list but not allow list
  return(true);			        // block
}

WORD CALLBACK HandleEventHook(RFTPEventStr* pEventStruc)
// ******
// Event (hook) handler.
// ******
{
  WORD Ret=REVNT_Proceed;               // return value, default is to allow operation
  char* pName=NULL;                     // pointer to path/file name being uploaded/renamed

  // disect event
  switch (pEventStruc->Event) {
    case EVNT_HookUp:                   // catch uploads
    case EVNT_HookAppend:               // same for APPE and STOU commands
    case EVNT_HookUnique:
      pName=pEventStruc->AuxOne;        // set pointer to file name
      break;
    case EVNT_HookRename:               // catch renames
      pName=pEventStruc->AuxTwo;
      break;
  }

  // process event when needed
  if (pName) {
    if (CheckFilterFile(pName)) {         // check if name matches list of blocked names
      pEventStruc->pReplyText=BLOCKREPLY; // set (static) FTP return message
      Ret=REVNT_Abort;                    // block event
    }
  }

  return(Ret);
}

BOOL WINAPI DllEntryPoint(HINSTANCE hInstance,DWORD fdwReason,PVOID /*pvReserved*/)
// ******
// Window's DLL entry point.
// Used for initializing list of names-to-block.
// ******
{
  switch (fdwReason) {
    case DLL_PROCESS_ATTACH:
    case DLL_THREAD_ATTACH:
      LoadFilterList(hInstance);        // init block list
      LoadAllowList(hInstance);         // init allow list
      break;
  }

  return(TRUE);                         // signal success to Windows
}