#!/usr/bin/perl -w 
#
# Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
#
# status.pl
#
# This script returns the current power status (on, off, suspended) of the 
# virtual machine specified by config on the server defined by hostname.
#
# usage: 
#   status.pl <path_to_config_file> [<server> <user> <password>]
#
# If server, user and password are not given, connect to the local server
# as the current user.
#

BEGIN {
   if ($^O eq "MSWin32") {
      @INC = (
         # Set the path to your VmPerl Scripting directory if different
         'C:\Program Files\VMware\VMware VmPerl Scripting API\perl5\site_perl\5.005',
         'C:\Program Files\VMware\VMware VmPerl Scripting API\perl5\site_perl\5.005\MSWin32-x86');
   }
}

use VMware::VmPerl;
use VMware::VmPerl::VM;
use VMware::VmPerl::ConnectParams;
use strict;

# Retrieves a pre-defined constant value.
sub vm_constant {
   my $constant_str = shift;
   return VMware::VmPerl::constant($constant_str, 0);
}

if (@ARGV < 1) {
   print "Usage $0: <path_to_config_file> [<server> <user> <password>]\n";
   exit(1);
}

my $state_string_map = {};
my @state_strings = ( 
   "VM_EXECUTION_STATE_ON", 
   "VM_EXECUTION_STATE_OFF", 
   "VM_EXECUTION_STATE_SUSPENDED", 
   "VM_EXECUTION_STATE_STUCK", 
   "VM_EXECUTION_STATE_UNKNOWN" 
    );

foreach my $state_string (@state_strings) {
   $state_string_map->{vm_constant($state_string)} = $state_string;
}

# Read in parameters.
my ($cfg_path, $server_name, $user, $passwd) = @ARGV;

# Use the default port of 902.  Change this if your port is different.
my $port = 902;

my $connect_params = VMware::VmPerl::ConnectParams::new($server_name,$port,$user,$passwd);

my $vm = VMware::VmPerl::VM::new();
if (!$vm->connect($connect_params, $cfg_path)) {
   my ($error_number, $error_string) = $vm->get_last_error();
   die "Could not connect to vm: Error $error_number: $error_string\n";
}

# Get the power status of the virtual machine.
my $cur_state = $vm->get_execution_state();
if (!defined($cur_state)) {
   my ($error_number, $error_string) = $vm->get_last_error();
   die "Could not get execution state: Error $error_number: $error_string\n";
}
print "The execution state of $cfg_path is: $state_string_map->{$cur_state}\n";

# Destroys the virtual machine object, thus disconnecting from the virtual machine.
undef $vm;
