/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.jackrabbit.vault.util;

import java.util.function.Supplier;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
A supplier wrapper which caches the supplied values. This class is not threadsafe!
 * @param <T>
 * @since FileVault 4.0.2
 */

public class CachingSupplier<T> implements Supplier<T>{

    T resolvedValue = null;
    Supplier<T> delegate;
    
    private CachingSupplier (Supplier<T> supplier) {
        delegate = supplier;
    }
    
    public static<T> CachingSupplier<T> of(@NotNull Supplier<T> supplier) {
        return new CachingSupplier<T>(supplier);
    }

    /**
     * Uses the cached value from previous get() calls instead of contacting the underlying delegate
     * @return the resolved value (can be null);
     */
    public @Nullable T get() {
        if (resolvedValue == null) {
            resolvedValue = delegate.get();
        }
        return resolvedValue;
    }

}
