/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.myfaces.commons.resourcehandler;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.zip.GZIPOutputStream;

import javax.faces.FacesException;
import javax.faces.context.FacesContext;

import org.apache.myfaces.commons.resourcehandler.resource.ResourceLoader;
import org.apache.myfaces.commons.resourcehandler.resource.ResourceLoaderWrapper;
import org.apache.myfaces.commons.resourcehandler.resource.ResourceMeta;
import org.apache.myfaces.commons.resourcehandler.resource.ValueExpressionFilterInputStream;

/**
 * 
 * @author Leonardo Uribe
 * @author Jakob Korherr
 *
 */
public class GZIPResourceLoader extends ResourceLoaderWrapper
{
    
    public final static String COMPRESSED_FILES_MAP = "oam.commons.COMPRESSED_FILES_MAP";
    
    /**
     * Subdir of the ServletContext tmp dir to store compressed resources.
     */
    private static final String COMPRESSION_BASE_DIR = "oam-resourcehandler-cache/";

    /**
     * Suffix for compressed files.
     */
    private static final String COMPRESSED_FILE_SUFFIX = ".gzip";
    
    /**
     * Size of the byte array buffer.
     */
    private static final int BUFFER_SIZE = 2048;
    
    private ResourceLoader delegate;
    
    private volatile File _tempDir;
    
    private final ExtendedDefaultResourceHandlerSupport _extendedDefaultResourceHandlerSupport;
    
    public GZIPResourceLoader(ResourceLoader delegate, ExtendedDefaultResourceHandlerSupport extendedDefaultResourceHandlerSupport)
    {
        this.delegate = delegate;
        _extendedDefaultResourceHandlerSupport = extendedDefaultResourceHandlerSupport;
        initialize();
    }
    
    protected void initialize()
    {
        //Get startup FacesContext
        FacesContext facesContext = FacesContext.getCurrentInstance();
    
        //1. Create temporal directory for compressed resources
        Map<String, Object> applicationMap = facesContext.getExternalContext().getApplicationMap();
        File tempdir = (File) applicationMap.get("javax.servlet.context.tempdir");
        File imagesDir = new File(tempdir, COMPRESSION_BASE_DIR);
        if (!imagesDir.exists())
        {
            imagesDir.mkdirs();
        }
        else
        {
            //Clear the cache
            deleteDir(imagesDir);
            imagesDir.mkdirs();
        }
        _tempDir = imagesDir;

        //2. Create map for register compressed resources
        Map<String, FileProducer> compressedFilesMap = new ConcurrentHashMap<String, FileProducer>();
        facesContext.getExternalContext().getApplicationMap().put(COMPRESSED_FILES_MAP, compressedFilesMap);
    }

    private static boolean deleteDir(File dir)
    {
        if (dir.isDirectory())
        {
            String[] children = dir.list();
            for (int i = 0; i < children.length; i++)
            {
                boolean success = deleteDir(new File(dir, children[i]));
                if (!success)
                {
                    return false;
                }
            }
        }
        return dir.delete();
    }
    
    @Override
    public URL getResourceURL(ResourceMeta resourceMeta)
    {
        FacesContext facesContext = FacesContext.getCurrentInstance();

        if (!_extendedDefaultResourceHandlerSupport.isCompressable(resourceMeta) || !_extendedDefaultResourceHandlerSupport.userAgentSupportsCompression(facesContext))
        {
            return super.getResourceURL(resourceMeta);
        }
        
        if (resourceExists(resourceMeta))
        {
            File file = createOrGetCompressedFile(facesContext, resourceMeta);
            
            try
            {
                return file.toURL();
            }
            catch (MalformedURLException e)
            {
                throw new FacesException(e);
            }
        }
        else
        {
            return null;
        }
    }    
    
    @Override
    public InputStream getResourceInputStream(ResourceMeta resourceMeta)
    {
        FacesContext facesContext = FacesContext.getCurrentInstance();

        if (!_extendedDefaultResourceHandlerSupport.isCompressable(resourceMeta) || !_extendedDefaultResourceHandlerSupport.userAgentSupportsCompression(facesContext))
        {
            return super.getResourceInputStream(resourceMeta);
        }
            
        if (resourceExists(resourceMeta))
        {
            File file = createOrGetCompressedFile(facesContext, resourceMeta);
            
            try
            {
                return new BufferedInputStream(new FileInputStream(file));
            }
            catch (FileNotFoundException e)
            {
                throw new FacesException(e);
            }
        }
        else
        {
            return null;
        }
    }
    
    @Override
    public boolean resourceExists(ResourceMeta resourceMeta)
    {
        return super.resourceExists(resourceMeta);
    }

    @SuppressWarnings("unchecked")
    private File createOrGetCompressedFile(FacesContext facesContext, ResourceMeta resourceMeta)
    {
        String identifier = resourceMeta.getResourceIdentifier();
        File file = getCompressedFile(resourceMeta);
        if (!file.exists())
        {
            Map<String, FileProducer> map = (Map<String, FileProducer>) 
                facesContext.getExternalContext().getApplicationMap().get(COMPRESSED_FILES_MAP);

            FileProducer creator = map.get(identifier);
            
            if (creator == null)
            {
                synchronized(this)
                {
                    creator = map.get(identifier);
                    
                    if (creator == null)
                    {
                        creator = new FileProducer();
                        map.put(identifier, creator);
                    }
                }
            }
            
            if (!creator.isCreated())
            {
                creator.createFile(facesContext, resourceMeta, file, this);
            }
        }
        return file;
    }    
    
    private File getCompressedFile(ResourceMeta resourceMeta)
    {
        return new File(_tempDir, resourceMeta.getResourceIdentifier() + COMPRESSED_FILE_SUFFIX);
    }

    private boolean couldResourceContainValueExpressions(ResourceMeta resourceMeta)
    {
        return resourceMeta.couldResourceContainValueExpressions() || resourceMeta.getResourceName().endsWith(".css");
    }
    
    
    /**
     * Uses GZIPOutputStream to compress this resource.
     * It will be stored where getCompressedFile() points to.
     *
     * Note that the resource really must be compressable (isCompressable() must return true).
     *
     * @return
     */
    protected void createCompressedFileVersion(FacesContext facesContext, ResourceMeta resourceMeta, File target)
    {
        //File target = getCompressedFile(resourceMeta);
        target.mkdirs();  // ensure necessary directories exist
        target.delete();  // remove any existing file

        InputStream inputStream = null;
        FileOutputStream fileOutputStream;
        GZIPOutputStream gzipOutputStream = null;
        try
        {
            if (couldResourceContainValueExpressions(resourceMeta))
            {
                inputStream = new ValueExpressionFilterInputStream(
                        getWrapped().getResourceInputStream(resourceMeta),
                        resourceMeta.getLibraryName(), 
                        resourceMeta.getResourceName());
            }
            else
            {
                inputStream = getWrapped().getResourceInputStream(resourceMeta);
            }
            fileOutputStream = new FileOutputStream(target);
            gzipOutputStream = new GZIPOutputStream(fileOutputStream);
            byte[] buffer = new byte[BUFFER_SIZE];

            pipeBytes(inputStream, gzipOutputStream, buffer);
        }
        catch (FileNotFoundException e)
        {
            throw new FacesException("Unexpected exception while create file:", e);
        }
        catch (IOException e)
        {
            throw new FacesException("Unexpected exception while create file:", e);
        }
        finally
        {
            if (inputStream != null)
            {
                try
                {
                    inputStream.close();
                }
                catch (IOException e)
                {
                    // Ignore
                }
            }
            if (gzipOutputStream != null)
            {
                // also closes fileOutputStream   
                try
                {
                    gzipOutputStream.close();
                }
                catch (IOException e)
                {
                    // Ignore
                }
            }
        }
    }
    
    /**
     * Reads the specified input stream into the provided byte array storage and
     * writes it to the output stream.
     */
    private static void pipeBytes(InputStream in, OutputStream out, byte[] buffer) throws IOException
    {
        int length;

        while ((length = (in.read(buffer))) >= 0)
        {
            out.write(buffer, 0, length);
        }
    }
    
    public static class FileProducer {
        
        public volatile boolean created = false;
        
        public FileProducer()
        {
            super();
        }

        public boolean isCreated()
        {
            return created;
        }

        public synchronized void createFile(FacesContext facesContext, ResourceMeta resourceMeta, File file, GZIPResourceLoader loader)
        {
            if (!created)
            {
                loader.createCompressedFileVersion(facesContext, resourceMeta, file);
                created = true;
            }
        }
    }
    
    public ResourceLoader getWrapped()
    {
        return delegate;
    }
}
