/*
 * SetTemplate.java
 *
 * Brazil project web application toolkit,
 * export version: 2.1 
 * Copyright (c) 2000-2004 Sun Microsystems, Inc.
 *
 * Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License Version 
 * 1.0 (the "License"). You may not use this file except in compliance with 
 * the License. A copy of the License is included as the file "license.terms",
 * and also available at http://www.sun.com/
 * 
 * The Original Code is from:
 *    Brazil project web application toolkit release 2.1.
 * The Initial Developer of the Original Code is: suhler.
 * Portions created by suhler are Copyright (C) Sun Microsystems, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s): cstevens, suhler.
 *
 * Version:  2.7
 * Created by suhler on 00/10/25
 * Last modified by suhler on 04/11/03 08:36:05
 */

package sunlabs.brazil.template;

import java.io.Serializable;
import java.util.Dictionary;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Properties;
import java.util.Vector;
import java.util.StringTokenizer;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;

import sunlabs.brazil.server.Handler;
import sunlabs.brazil.server.Request;
import sunlabs.brazil.server.Server;
import sunlabs.brazil.session.SessionManager;
import sunlabs.brazil.util.Format;
import sunlabs.brazil.util.Glob;
import sunlabs.brazil.util.http.HttpUtil;

/**
 * Template (and handler) class for setting and getting values to
 * and from the current (or other) request context.
 * This class is used by the TemplateHandler.  The following tags
 * are processed by this class:
 * <ul>
 * <li>&lt;set name=.. [value=..] [namespace=..]&gt;
 * <li>&lt;get name=.. [default=..] [namespace=..] [max=nnn] [convert=html|url|none] [set=..]&gt;
 * <li>&lt;namespace name=.. [clear] [remove] [store=xxx] [load=xxx]&gt;
 * <li>&lt;import namespace=.. &gt;
 * <li>&lt;tag [name=..]&gt;
 * <li>&lt;/tag&gt;
 * </ul>
 * <p>
 * The tag<code>&lt;set&gt;</code> is processed with the following
 * attributes:
 * <dl class=attributes>
 * <dt> name=<i>value</i>
 * <dd> The name of the entry to set in the request properties.
 *
 * <dt> value=<i>value</i>
 * <dd> The value to store in the property.  If this option is not specified,
 *	then the existing value for this property will be removed.
 *
 * <dt> default=<i>other</i>
 * <dd>	If <code>default</code> is specified and the specified <i>value</i>
 *	was the empty string "", then store this <i>other</i> value in the
 *	property instead.  <i>other</i> is only evaluated in this case.
 *
 * <dt> local
 * <dd> By default, variables are set (or removed from) a namespace that is
 *	persistent across all requests for the same session. If
 *	<code>local</code> is specified, the values only remain for the
 *	duration of the current request. [Deprecated - use namespace=local]
 *
 * <dt> namespace
 * <dd> The namespace to use to set or delete the property.  The default
 *	is <code>namespace=${SessionID}</code>, unless <code>set</code> is
 *      inside the scope of an enclosing <code>namespace</code> tag.
 *      Namespaces are used to define
 *	variable scopes that may be managed explicitly in template pages.
 *	<p>
 *	Two namespaces are treated specially:
 *	<ul>
 *	<li><b>local</b><br>
 *	may be used to set (or clear) values
 *	from <code>request.props</code>.
 *	<li><b>server</b><br>
 *	may be used to set (or clear) values
 *	from <code>server.props</code>, unless "noserver" is defined
 *	in which case hte "Server" namespace is not treated specially.
 *      </ul>
 * </dl>
 * <p>
 * The <code>&lt;namespace&gt;</code>..<code>&lt;/namespace&gt;</code> tag
 * pairs are used to specify a default namespace other than the 
 * <code>SessionId</code> for the <b>set</b> and <b>import</b> tags
 * or manipulate a namespace.
 * These tags don't nest.
 * <p>
 * attributes for <code>namespace</code>:
 * <dl class=attributes>
 * <dt> name=<i>value</i>
 * <dd> The default namespace to use for the <b>set</b> and <code>import</code>
 *	default namespace name.
 * <dt> clear
 * <dd> If set, namespace <code>name</code> is cleared.
 * <dt> remove
 * <dd> If set, namespace <code>name</code> is removed.
 * <dt> load=<i>file_name</i>
 * <dd> If specified, and the property <code>saveOk</code> is set, then
 *	the specified namespace will be saved loaded from a file in java
 *      properties
 *	format.  Relative path names are resolved with respect to the
 *	document root.
 * <dt> store=<i>file_name</i>
 * <dd> If specified, and the property <code>saveOk</code> is set, then
 *	the specified namespace will be stored to a file in java properties
 *	format.  Relative path names are resolved with respect to the
 *	document root.
 * </dl>
 * When using <code>load</code> or <code>store</code>, the namespace
 * names "local" and "server" are treated specially, referring to
 * "request.props" and "server.props" respectively.
 * <p>
 * The tag<code>&lt;import&gt;</code> is processed with the following
 * attributes:
 * <dl class=attributes>
 * <dt> namespace=<i>value</i>
 * <dd> The name/value pairs assiciated with the namespace are
 *	made available for the remainder of the page.
 * </dl>
 * <p>
 * The "get" (formerly property) tag has the following attributes:
 * <dl>
 * <dt>name
 * <dd>The name of the variable to get
 * <dt>namespace
 * <dd>The namespace to look in.  By default, the variable is searched
 *     for in "request.props"
 * <dt>default
 * <dd>The value to use if no value matches "name".
 * <dt>convert
 * <dd>The conversion to perform on the value of the data before
 *     substitution: "html", "url", or "none" (the default). For
 *     "html", any special html syntax is escaped.
 *     For "url", the data will be suitable for transmission as an
 *     http URL.
 * <dt>max
 * <dd>The output is truncated to at most <code>max</code> characters.
 * </dl>
 * If a single attribute is specified, with no "=", then is is taken
 * to be the "name" parameter.  Thus:
 * <code>&lt;get foo&gt;</code> is equivalent to:
 * <code>&lt;get name="foo"&gt;</code>.
 * <p>
 * Request Properties:
 * <dl class=props>
 * <dt>sessionTable
 * <dd>The name of the SessionManager table to use for
 *     storing values.  Defaults to the handler's prefix.  When configured
 *     in one server both as a handler and a template, the sessionTable should
 *     be set to the same value in both cases to get a useful result.
 * <dt>debug
 * <dd>If set, the original tag is included in a comment, 
 *     otherwise it is removed entirely.
 * <dt>mustMatch
 * <dd>Set to a glob pattern that all names must match
 *     in order to be set.  This may be used to prevent malicious
 *     html pages (what a concept) from changing inappropriate values.
 * <dt>noserver
 * <dd>The "server" namespace will no longer be mapped to
 *	<code>server.props</code>
 * <dt>noSet
 * <dd>If set, then the "set" tag will be disabled.
 * <dt>querySet
 * <dd>If set, then properties may be set in query 
 *     parameters, to the "handler" portion, but only
 *     if they match the glob pattern.
 * <dt>autoImport
 * <dd>If set to "1", the namespace for the session is 
 *     automatically imported. (defaults to "1");
 * <dt>imports
 * <dd>Defines a set of (white space delimited) namespaces that will
 *     automatically be imported at the beginning of each page.
 *     Each namespace name will be processed for ${...} substitutions before
 *     an import is attempted.  If the namespace doesn't already exist, 
 *     the import is ignored.
 * <dt>query	<dd> The query parameters are inserted into the request object,
 *	        prefixed by the value assigned to "query".
 * <dt>headers	<dd> The mime headers are inserted into the request object,
 *		prefixed by the value assigned to "headers".
 *              <p>In addition, the following properties (prefixed with
 *		"headers") are also set:
 *              <ul>
 *		<li><b>address</b> The ip address of the client
 *		<li><b>counter</b> A monotonically increasing counter
 *		    (# of client requests accepted since the server started).
 *		<li><b>method</b> The request method (typically GET or POST).
 *		<li><b>protocol</b> The client HTTP protocol level (for 1.0 or 1.1)
 *		<li><b>query</b> The current query data, if any.
 *		<li><b>timestamp</b> A timestamp (in ms since epoch) from when
 *			this request was first accepted.
 *		<li><b>url</b> The current url.
 *              </ul>
 * <dt>url.orig	<dd> If set and "headers" are requested, this value is used as
 *		the <code>url</code> instead of the one in request.url.
 * </dl>
 * <p>
 * Normally, any persistent properties held by the SetTemplate are 
 * chained onto the request.props when the init method is found.  If this
 * template is installed as an up-stream handler, then the persistent
 * properties associated with the session are made available at that time.
 * <p>
 * When used as a handler, the following property is used:
 * <dl class=props>
 * <dt>session=<i>value</i>
 * <dd>The request property to find the session information in.
 *     Normally this should be the same as the session property used
 *     by the container calling this as a template.
 * <dt>saveOk
 * <dd>This must be specified in order for the
 *     "namespace store" or "namespace load" functions to operate.
 * </dl>
 * 
 * @author		Stephen Uhler
 * @version		@(#)SetTemplate.java	2.7
 */

public class SetTemplate extends Template implements Serializable, Handler {
    String mustMatch;		// glob-pattern of permitted names
    String prefix = null;	// non-null if instance is a handler
    String session;		// the handler property for the session var
    String sessionTable;	// the "other" session key
    String querySet;		// if non-null set values from query
    boolean allowServer=true;	// allow sets to server properties
    boolean noSet = false;	// if true, "set" is disabled"
    String current;		// current namespace 

    Vector remember = new Vector();	// future chainable namespaces

    /**
     * Chain the session-id properties into the request chain, if
     * there are any.  If "query" or "headers" are requested for "get", 
     * then add in those properties to request.props.
     */

    public boolean
    init(RewriteContext hr) {
	Properties props = hr.request.props;
	current = null;
	String query = props.getProperty(hr.prefix + "query");
	if (query != null) {
	    Dictionary h = hr.request.getQueryData(null);
	    Enumeration keys = h.keys();
	    while(keys.hasMoreElements()) {
		String key = (String) keys.nextElement();
		props.put(query + key, h.get(key));
	    }
	}
	String headers = props.getProperty(hr.prefix + "headers");
	if (headers != null) {
	    Enumeration keys = hr.request.headers.keys();
	    while(keys.hasMoreElements()) {
		String key = (String) keys.nextElement();
		props.put(headers + key.toLowerCase(),
			hr.request.headers.get(key));
	    }
	    props.put(headers + "method", hr.request.method);
	    if (props.containsKey("url.orig")) {
		props.put(headers + "url", props.getProperty("url.orig"));
	    } else {
		props.put(headers + "url", hr.request.url);
	    }
	    props.put(headers + "query", hr.request.query);
	    props.put(headers + "protocol", hr.request.protocol);
    	    props.put(headers + "address",
		"" + hr.request.getSocket().getInetAddress().getHostAddress());
	    props.put(headers + "timestamp", "" + hr.request.startMillis);
	    props.put(headers + "counter", "" + hr.server.acceptCount);
	}

	remember.removeAllElements();
	mustMatch = hr.request.props.getProperty(hr.prefix + "mustMatch");
	allowServer = (null == hr.request.props.getProperty(hr.prefix +
		"noserver"));
	noSet = (null != hr.request.props.getProperty(hr.prefix +
		"noSet"));
	sessionTable = hr.request.props.getProperty(hr.prefix + "sessionTable",
		hr.prefix);

	String autoImport = hr.request.props.getProperty(hr.prefix +
	    "autoImport", "1");
	if  (autoImport.equals("1")) {
	    Properties p = (Properties) SessionManager.getSession(hr.sessionId,
		    sessionTable, null);
	    if (p != null) {
		hr.request.addSharedProps(p);
	    } else {
	        remember.addElement(hr.sessionId);
		hr.request.log(Server.LOG_DIAGNOSTIC, hr.prefix,
			"Remembering (init) " + sessionTable);
	    }
	}

	String imports = hr.request.props.getProperty(hr.prefix + "imports");
	if (imports != null) {
	    StringTokenizer st = new StringTokenizer(imports);
	    while (st.hasMoreTokens()) {
	        String token = Format.subst(hr.request.props, st.nextToken());
	        Properties p = (Properties) SessionManager.getSession(token,
		    sessionTable, null);
	        if (p != null) {
		    hr.request.addSharedProps(p);
		    hr.request.log(Server.LOG_DIAGNOSTIC, hr.prefix,
			    "Auto importing: " + token);
		}
	    }
	}
	return super.init(hr);
    }

    /**
     * Set the value of a variable.  Allow variable substitutions in the name
     * and value.  Don't create property tables needlessly.
     * <p>
     * Attributes:
     * <dl>
     * <dt>name
     * <dd>The name of the variable to set
     * <dt>value
     * <dd>The value to set if to.
     * <dt>namespace
     * <dd>The namespace to look in.  By default, the variable is set in
     *     the namespace associated with the current "SessionId".
     * <dt>local
     * <dd>A (deprecated) alias for "namespace=local", or the current
     *     request.props.
     * </dl>
     */

    public void
    tag_set(RewriteContext hr) {
	if (noSet) {
	    return;
	}
        debug(hr);
	hr.killToken();

	String name =  hr.get("name");
	if (name == null) {
	    debug(hr, "missing variable name");
	    return;
	}
	if (mustMatch!=null && !Glob.match(mustMatch, name)) {
	    debug(hr, name + " doesn't match " + mustMatch);
	    return;
	}
	String value =  hr.get("value");
	String namespace =  hr.get("namespace");
	boolean local = (hr.isTrue("local") || "local".equals(namespace));
	Class create  = (value != null)  ? Properties.class : null;

	if (namespace == null) {
	    namespace = (current == null) ? hr.sessionId : current;
	}

	/*
	 * Find the correct property table.  If we are
	 * clearing an entry, don't make the table if it doesn't exist.
	 */

	Properties p;	// table to set or clear
	if (local) {
	    p = hr.request.props;
	    namespace="local";
	} else if (allowServer && "server".equals(namespace)) {
	    p = hr.server.props;
	} else  {
	    p = (Properties) SessionManager.getSession(namespace,
		sessionTable, create);
	}

	// set or clear the entry

	if (value != null) {
	    if (value.equals("")) {
		String dflt = hr.get("default");
		if (dflt != null) {
		    value = dflt;
		}
	    }
	    if (p.size()==0 && remember.contains(namespace)) {
		hr.request.addSharedProps(p);
		remember.removeElement(p);
		hr.request.log(Server.LOG_DIAGNOSTIC, hr.prefix,
			"Importing " + namespace);
	    }
	    p.put(name, value);
	    debug(hr, namespace + "(" + name + ") = " + value);
	} else if (p != null) {
	    p.remove(name);
	}
    }

    /**
     * Convert the html tag "property" in to the request's property
     * DEPRECATED - use "get"
     */

    public void
    tag_property(RewriteContext hr) {
        tag_get(hr);
    }

    /**
     * Replace the tag "get" with the value of the
     * variable specified by the "name" attribute.
     * <p>
     * Attributes:
     * <dl>
     * <dt>name
     * <dd>The name of the variable to get
     * <dt>namespace
     * <dd>The namespace to look in.  By default, the variable is searched
     *     for in "request.props".  The namespace "server" is used to look in
     *	   the server's namespace.  The namespace "local" is a synonym for the
     *	   default namespace.
     * <dt>default
     * <dd>The value to use if no value matches "name".
     * <dt>convert
     * <dd>The conversion to perform on the value of the data before
     *     substitution: "html", "url", or "none" (the default). For
     *     "html", any special html syntax is escaped.
     *     For "url", the data will be suitable for transmission as an
     *     http URL.
     * <dt>max
     * <dd>Truncate the String to at most <code>max</code> characters.
     *     Max must be at least one, and truncation occurs after
     *     any conversions.
     * <dt>set
     * <dd>The resultant value is placed into the request property named by
     *     the <code>set</code> attribute, and not inserted into the
     *     HTML stream.  If none of "namespace", "convert", or "match"
     *	   is used, then this simply copies the property from one name
     *     to another.
     * </dl>
     * If a single attribute is specified, with no "=", then is is taken
     * to be the "name" parameter.  Thus:
     * <code>&lt;get foo&gt;</code> is equivalent to:
     * <code>&lt;get name="foo"&gt;</code>.
     */

    public void
    tag_get(RewriteContext hr) {
	String name = hr.getArgs();
	String result = null;
	if (name.indexOf('=') >= 0) {
	    name = hr.get("name");
	    result = hr.get("default");
	} else {
	    name = Format.subst(hr.request.props, name);
	}

	if (name == null) {
	    debug(hr, "get: no name");
	    return;
	}

	String namespace = hr.get("namespace");
	if (namespace != null) {
	    Properties p;
	    if (namespace.equals("server")) {
		p = hr.server.props;
	    } else if (namespace.equals("local")) {
		p = hr.request.props;
	    } else {
	        p = (Properties) SessionManager.getSession(namespace,
		    sessionTable, null);
	    }
	    result = p==null ? result : p.getProperty(name, result);
	} else {
	    result = hr.request.props.getProperty(name, result);
	}

	if (result != null) {
	    String convert = hr.get("convert");
	    if (convert != null) {
	        if (convert.indexOf("html")>=0) {
		    result = HttpUtil.htmlEncode(result);
		}
	        if (convert.indexOf("url")>=0) {
		    result = HttpUtil.urlEncode(result);
	        }
	        if (convert.indexOf("lower")>=0) {
		    result = result.toLowerCase();
	        }
	    }
	    int max = 0;
	    try {
		String str = hr.get("max");
		max = Integer.decode(str).intValue();
	    } catch (Exception e) {}
	    if (max > 0 && max < result.length()) {
		result = result.substring(0,max);
	    }

	    String set = hr.get("set");  // set to value only 
	    if (set != null) {
	       hr.request.props.put(set, result);
	    } else {
	       hr.append(result);
	    }
	} else {
	    debug(hr, "property: no value for " + name);
	}
	hr.killToken();
    }

    /**
     * Import all the data from the named namespace.  The namespace
     * associated with the session ID is imported automatically
     * for backward compatibility.  If the namespace doesn't exist, 
     * don't create it now, but remember it needs to be "Chained"
     * if it is created on this page.
     */

    public void
    tag_import(RewriteContext hr) {
        debug(hr);
	hr.killToken();
	String namespace =  hr.get("namespace");
	if (namespace == null && current != null) {
	    namespace = current;
	    // System.out.println("Importing namespace: " + namespace);
	}
	if (namespace != null) {
	    Properties p = (Properties) SessionManager.getSession(namespace,
		    sessionTable, null);
	    if (p != null) {
	        hr.request.addSharedProps(p);
	    } else if (!remember.contains(namespace)){
	        remember.addElement(namespace);
		hr.request.log(Server.LOG_DIAGNOSTIC, hr.prefix,
			"Remembering " + namespace);
	    }
	}
    }

    /**
     * Set the default namespace for "set" and "import".
     */

    public void
    tag_namespace(RewriteContext hr) {
	String name = hr.get("name");
        debug(hr);
	hr.killToken();
	if (name != null) {
	    if (!hr.isSingleton()) {
	        current = name;
	    }
	    String load = hr.get("load");
	    String store = hr.get("store");
	    boolean ok =
		  (null != hr.request.props.getProperty(hr.prefix + "saveOk"));

	    // XXX should dis-allow ambiguous options

	    Properties p;	// which property to load or set
		boolean isSm = false;	// true if session manager table
		if (name.equals("local")) {
			p = hr.request.props;
		} else if (name.equals("server")) {
			p = hr.server.props;
		} else {
			p = (Properties) SessionManager.getSession(name,
				sessionTable, null);
			isSm = true;
		}

	    if (ok && store != null && p != null) {
		   store(p, hr, store, name);
	    }

	    if (ok && load != null) {
		if (p == null) {
		    p = (Properties) SessionManager.getSession(name,
			sessionTable, Properties.class);
		}
		load(p, hr, load);
	    }

	    if (isSm && hr.isTrue("clear") && p != null) {
		p.clear();
	    }
	    if (isSm && hr.isTrue("remove") && p != null) {
		SessionManager.remove(name, sessionTable);
		p = null;
	    }
	} else {
	    debug(hr, "No namespace to import");
	}
    }
 
    /**
     * Convert a file name into a file object.
     * relative paths use are resolved relative to the document root
     * @param p		properties file to find the doc root in
     * @param prefix	the properties prefix for root
     * @param name	the name of the file
     */

    public File
    file2path(Properties p, String prefix, String name) {
	File file;
	if (name.startsWith("/")) {
	    return new File(name);
	} else {
	    String root = p.getProperty(prefix + "root", 
		    p.getProperty("root", "."));
	    return new File(root, name);
	}
    }

    /**
     * load a namespace from a java properties file
     */

    boolean
    load(Properties p, RewriteContext hr, String name) {
	File file = file2path(hr.request.props, hr.prefix, name);
	hr.request.log(Server.LOG_DIAGNOSTIC, hr.prefix,
		"loading namespace " + file);
	try {
	    FileInputStream in = new FileInputStream(file);
	    p.load(in);
	    in.close();
	    return true;
	} catch (IOException e) {
	    hr.request.log(Server.LOG_LOG, hr.prefix,
		"Can't load namespace " + file);
	    return false;
	}
    }

    /**
     * Store a namespace into a java properties file
     */

    boolean
    store(Properties p, RewriteContext hr, String name, String title) {
	File file = file2path(hr.request.props, hr.prefix, name);
	try {
	    FileOutputStream out = new FileOutputStream(file);
	    p.save(out, hr.request.serverUrl() + hr.request.url +
		" (for namespace " + title + ")");
	    out.close();
	    hr. request.log(Server.LOG_DIAGNOSTIC, hr.prefix,
		"Saving namespace " + title + " to " + file);
	    return true;
	} catch (IOException e) {
	    hr. request.log(Server.LOG_WARNING, hr.prefix,
		    "Can't save namespace " + title + " to " + file);
	    return false;
	}
    }

    /**
     * Clear the default namespace for "set" and "import".
     */

    public void
    tag_slash_namespace(RewriteContext hr) {
        debug(hr);
	hr.killToken();
	current = null;
    }

    /**
     * Map a set of variables from one name to another.
     * <dl class=attributes>
     * <dt>namespace 
     * <dd>The namespace of the destination (defaults to current namespace)
     * <dt>match
     * <dd>The regular expression that matches variable names in the
     *	   current context
     * <dt>replace
     * <dd>The substitution expression used to derice the new names
     * <dt>convert
     * <dd>Which conversions to perform on the data.  May be
     *	   "html", "url", or "none".  The default is "none"
     * <dt>clobber
     * <dd>"true" or "false".  If "false" (the default) existing variables
     *     will not be overridden
     * <dt>clear
     * <dd>"true" or "false".  If "true" (the default) the source variable
     *     will be deleted
     * </dl>
     */

/* Not done!

    public void
    tag_map(RewriteContext hr) {
        debug(hr);
	hr.killToken();
	if (noSet) {
	    return;
	}
	String match =  hr.get("match");
	String replace =  hr.get("replace");
	String namespace =  hr.get("namespace");
	String convert = hr.get("convert");

	if (match==null || replace == null) {
	    debug(hr, "missing \"match\" or \"replace\"");
	    return;
	}
	boolean local = (hr.isTrue("local") || "local".equals(namespace));

	if (namespace == null) {
	    namespace = (current == null) ? hr.sessionId : current;
	}

	// XXX move down
	if ("html".equals(convert)) {
	    result = HttpUtil.htmlEncode(result);
	} else if ("url".equals(convert)) {
	    result = HttpUtil.urlEncode(result);
	}
    }
*/

    /**
     * Insert a literal "&lt;".
     * Using the current scheme, there is no easy way to substitute into
     * a tag parameter.  So we'll invent a "magic" tag (called tag)
     * that will allow us to create entities dynamically.  Thus values
     * can be substituted into entities by escaping the entity as in:
     * <pre>
     * &lt;tag&gt;a href=&lt;property href&gt;&lt;/tag&gt;
     * </pre>
     * <p>
     * The [optional] attribute "name" may be used to specify the
     * name of the tag, which will be emmitted just after the "&lt;".
     */

    public void
    tag_tag(RewriteContext hr) {
	String name = hr.get("name");
	hr.append("<");
	if ( name != null) {
	    hr.append(name);
	}
    }

    /**
     * Insert a literal "&gt;"
     */

    public void
    tag_slash_tag(RewriteContext hr) {
	hr.append(">");
    }

    /**
     * Get the 2 SessionManager keys, "sessionTable" and "session" (SessionID).
     */

    public boolean
    init(Server server, String prefix) {
	this.prefix = prefix;
	querySet = server.props.getProperty(prefix + "querySet");
	sessionTable = server.props.getProperty(prefix + "sessionTable",
		prefix);
	session=server.props.getProperty(prefix + "session", "SessionID");
	return true;
    }

    /**
     * Chain a SessionManager entries onto the request properties, 
     * and optionally allow setting of request props from query parameters.
     * Only the session id table can be chained, and 
     * values (if any) are set in request.props (i.e. namespace=local)
     */

    public boolean
    respond(Request request) {
	String id = request.props.getProperty(session);
	if (id != null) {
	    Properties p = (Properties) SessionManager.getSession(id,
		    sessionTable, null);
	    if (p != null) {
	        request.addSharedProps(p);
		request.log(Server.LOG_DIAGNOSTIC, prefix,
			"Chaining: " + id + "," + sessionTable);
	    } else {
		request.log(Server.LOG_DIAGNOSTIC, prefix,
			"No table: " + id + "," + sessionTable);
	    }
	}

	/*
	 * Set some properties as part of the request.
	 * Should they be persistent or not?
	 */

	if (querySet!=null) {
	    Dictionary h = request.getQueryData(null);
	    Enumeration keys = h.keys();
	    while(keys.hasMoreElements()) {
		String key = (String) keys.nextElement();
	        if (Glob.match(querySet, key)) {
		    request.props.put(key, h.get(key));
		}
	    }
	}
	return false;
    }
}
