//
//  Copyright (c) 1999,2000 by Network TeleSystems, Inc.
//  All rights reserved.
//  Internet http://www.nts.com
//
//	Please direct all support-related calls to your licensed Internet Service 
//	Provider.  NTS does not provide direct support.  NTS works through Internet 
//	Service Providers that have licensed our software.
//

/*
	Method - This is user mode code that converts Async or Sync HDLC-framed
	PPP frames from the PPP stack via pppd/ptys to PPPoE frames.
	These are then passed back to the kernel to be sent out the ethernet
	port via raw sockets API.
 */

/*
Changes/fixes:  

001.100.001	
	1) If the AC-Name given on the command line was longer than the one in 
	   the packet a match may have been claimed incorrectly. If the AC-Name 
	   in the packet matched the beginning of that given on the command line.

	2) When it came to returning Relay-Session-Id and/or AC-Cookie
	   PPPoE_SendPADR() was busted.  The packet length was fixed.
	   tag-types were byte swapped and the wrong tag-type was being used for 
	   Relay-Session-Id.

	3) Reduced buffer sizes where they did not need to exceed 1514.
	   Got rid of some needless copy steps.
	
	4) Add retransmit logic.
	
	5) Added support for specifying service name. eg. -I ServiceNameHere
	
	6) If unused PADO responses have Relay-Session-Id/AC-Cookie
	   and the PADO response we use, does not. Then our reply PADR would carry
	   the Relay-Session-Id/AC-Cookie's that we had last seen from the other
	   PADO's.  There was also a memory leak in this area as well.
	   
	7) Check packets for error tags.  Previous would have taken an error
	   reply as a good response.  
	   
	8) Added printing to stderr for when error packets are delivered.
	
	9) Added a -L option to the command line.  This will display the responses
	   received to a PADI packet.

001.200.001	
   10) Moved away from doing standard IO, now use full buffer read/writes
	   against pty port.
	   
   11) The selected pty is passed as an argument on the command line.
       eg. /dev/ptyp0  note, I will also accept /dev/ttyp0 and convert it
	   to /dev/ptyp0.
	   
   12) Took a suggestion from the book, "UNIX Network Programming Vol. 1" by 
       W. R. Stevens. Now use handle from pcap to do sending of packets on 
	   ethernet.  It allows us to remove libnet from the build. Not much of 
	   the libary was being used.  Note, on some OSs the pcap fd is opened 
	   read only. This has to change to read/write.  This is not a problem for
	   Linux
	   
   13) Became selective about when to escape control characters going to pppd.
   
   14) Added Host-Uniq tag to PADI and PADR and checking for it on PADS and 
       PADO.  Ignore PADSs and PADOs that do not match.
	   
   15) There have been more changes than I've been able to keep track of.

001.201.001	
   16) Disabled optimization on control character send to pppd.
   
001.202.001	
   17) Got rid of the pcap library.

   18) Removed temp thread used at startup to monitor startup.
   
   19) Now using syslog for info messages.
   
   20) Now support sync option used by pppd. Sync is auto-detected.
       requires n_hdlc.o module found in 2.2.x Linux kernel.

001.203.001	   
   21) Started catching signals for a more orderly shutdown

001.203.002	   
   22) Syslog the access concentrator we try to connect to and indicate
       the service name that we select if any.
	   
   23) Error tags are now writen to syslog instead of stderr.  Writing
       to stderr is now only done when performing a -L option or when error's
	   occur while processing command line options.
	   
   24) Corrected length passed to read function when reading into 
   	   "the_Frame".

001.203.003
   25) input_loop, async area, for() Messed up test of loop index in body
       of loop.  Needed to test against ++NextByte.

001.203.004
   26) Fixed start-pppoe -I option was missing when invoking pppoe.

   27) Updated Makefile to link with shared version of the pthread library 
       rather than static.  The static lib in the Mandrake 6.1 release would
       cause the app to hang.  Even simple pthread test programs would hang.
       This problem did not show in RedHat 6.0 or 6.1.

001.203.005
   28) Added lcp... options to options.pppoe file.  So a crashed concentrator
       can be timed out.  Updated stop-pppoe to use SIGINT, since SIGHUP would
       not stop pppd when the persist option is used.

001.203.006
   29) Improved the auto-detect algorithm for sync/async framing method.  
	 
   30) Added timeout logic to code that reads data from pppd, this works with 
	   LCP echo request that come from pppd to help close down pppoe when it 
	   has been abandoned by pppd.
	 
   31) Pppoe now watches for LCP echo terminate so it can learn that a 
	   connections is being shutdown.  
	 
   32) Command line errors are now placed in syslog and written to stderr.  
	 
   33) The pppoe -L option waits 4 seconds for all the responces to come back.  
	   Now if you press the enter key it will abort the 4 second timer and exit.

   34) The Auto-detect of sync/async framing method can now be overridden.
	   Add the "sync" option to the pppoe command line when the sync option is 
	   used with pppd.  Add the async option to the pppoe command line when 
       the sync option is not specified as a pppd command option.

   35) Added changes and conditionals for compiling sockets to use
       PF_PACKET/SOCK_RAW for Linux 2.2 vs. AF_INET/SOCK_PACKET for Linux 2.0.

001.203.008
   36) Merged main.c and pppoe-discovery.c into pppoe.c and pppoe.h etc.
       Broke input_loop() into ReadPppd, AutoDetectFrameMethod, AsyncPppdLoop, 
       SyncPppdLoop, Pppd2Ethernet

001.203.009
   37) Deleted commented out code.

001.300.000
   38) Rolled to version number for final release.  Made changes to indicate 
	   release candidate number with version information.   
	   All version 1.203.xxx were beta.

001.310.000
   39) Added logic to send PADT just before exit, if an active PPPoE session
       still exist.  Added 2 sec. delay on shutdown (SIGHUP or SIGINT) to 
       allow pppd to send and receive LCP:Terminate.  When a successful 
       LCP:Terminate-ACK has been seen or sent then no PADT is sent.
 */
#define MAJOR_VER (1)
#define MINOR_VER (310)
#define BUILD_NO (0)
#ifndef RELEASE_CANDIDATE
#define RELEASE_CANDIDATE ""
#endif

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <features.h>
#if __GLIBC__ >= 2 && __GLIBC_MINOR__ >= 1
#include <netpacket/packet.h>
#else
#include <asm/types.h>
#include <linux/if_packet.h>
#include <linux/if_ether.h>
#endif
#include <net/ethernet.h>
#include <ctype.h>
#include <pthread.h>
#include <signal.h>
#include <errno.h>

#include <net/if.h>
#include <net/if_arp.h>
#include <syslog.h>

#include "pppoe.h"

#ifndef N_HDLC
#define	N_HDLC	13
#endif

#ifndef PRIORITY_BOOST
#define PRIORITY_BOOST 75
#endif
int PriorityBoost = PRIORITY_BOOST;

#define INPUT_TIMEOUT_COUNT 12
#define INPUT_TIMEOUT_INTERVAL 10
#define INPUT_TIMEOUT_DELAY (INPUT_TIMEOUT_INTERVAL * INPUT_TIMEOUT_COUNT)

//#if (__linux__)

#define MAC_EQUAL( src, dest ) \
	( *((unsigned long  *)(src)) == *((unsigned long  *)(dest)) && \
	  *((unsigned short *)&((u_char *)(src))[4]) == \
	  *((unsigned short *)&((u_char *)(dest))[4])     )
						   
#define SET_MAC_ADDR( packet, mac )\
{\
	*((unsigned long  *)(packet)) = *((unsigned long  *)(mac));\
	*((unsigned short *)&((u_char *)(packet))[4]) = \
	*((unsigned short *)&((u_char *)(mac))[4]); \
}

#define MAX(a, b) ( ((a) > (b)) ? (a) : (b) )

#define SET_STATE( a ) (gState=(a))

int Debug = 0;

//#define DEBUG
#ifdef DEBUG
#define DEBUGOUT( a ) (fprintf( stderr, "\npppoe: " a "\n") )
#define DEBUGOUT2( a, b ) (fprintf( stderr, "\npppoe: " a "\n", b))
#define DEBUGOUT3( a, b, c ) (fprintf( stderr, "\npppoe: " a "\n", b, c))
#else
#define DEBUGOUT( a )
#define DEBUGOUT2( a, b )
#define DEBUGOUT3( a, b, c )
#endif

#define kDLC_FLAG ((UInt8)0x07E)
#define kESC ((UInt8)0x07D)
#define MAX_RELAY_ID_LENGTH		12			//as specified by RFC-2516 

enum
	{
	kPPPoE_Tag_EndOfList			= 0x0000,
	kPPPoE_Tag_ServiceName			= 0x0101,
	kPPPoE_Tag_AC_Name				= 0x0102,
	kPPPoE_Tag_Host_Uniq			= 0x0103,
	kPPPoE_Tag_AC_Cookie			= 0x0104,
	kPPPoE_Tag_Vendor				= 0x0105,
	kPPPoE_Tag_Relay_Session_ID		= 0x0110,
	kPPPoE_Tag_Service_Name_Error	= 0x0201,
	kPPPoE_Tag_AC_System_Error		= 0x0202,
	kPPPoE_Tag_Generic_Error		= 0x0203,	
	
	kPPPoE_Disc_PADI				= 0x009,
	kPPPoE_Disc_PADO				= 0x007,
	kPPPoE_Disc_PADR				= 0x019,
	kPPPoE_Disc_PADS				= 0x065,
	kPPPoE_Disc_PADT				= 0x0A7,
	
	kPPPoE_Version					= 0x11,
	
	kPPPoE_Tag_Last					= 0xFFFF
	};

enum
	{
	kPPPoE_Init_State,
	kPPPoE_SentPADI_State,
	kPPPoE_SentPADR_State,
	kPPPoE_PPP_State,
	kPPPoE_SentPADT_State,
	kPPPoE_ReceivedPADT_State,
	kPPPoe_GatherACNames,
	kPPPoE_LastState,
	};

typedef struct _PPPoEPayloadHeader {
	UInt8	version;
	UInt8	code;
	UInt16	SessionID;
	UInt16	length;
} PPPoEPayloadHeader, *PPPoEPayloadHeaderPtr;

typedef struct _PPPoESession {
	PPPoEPayloadHeader	payloadHeader;
	UInt8				data[0];
} PPPoESession, *PPPPoESession;

typedef struct _PPPoETag {
	UInt16	type;
	UInt16	length;
	UInt8	data[0];
} PPPoETag, *PPPoETagPtr;

#define TAG_TYPE_LENGTH_SIZE (sizeof(UInt16)+sizeof(UInt16))
#define NEXT_TAG_ITEM( a ) ( (a) = (PPPoETag *)\
		(((char *)(a)) +TAG_TYPE_LENGTH_SIZE +ntohs(((PPPoETag *)(a))->length)) )

typedef struct PPPoEPacket {
	PPPoEPayloadHeader	payloadHeader;
	PPPoETag	       	tag;
} PPPoEPacket, *PPPoEPacketPtr;


pthread_mutex_t gSendMutex	= PTHREAD_MUTEX_INITIALIZER;
pthread_cond_t  gSendCond	= PTHREAD_COND_INITIALIZER;

u_char 				gXmitFrame[1514];
UInt16				gXmitFrameLength;
u_char				bSendRequest;

struct ether_addr 	broadcast = {{0xff, 0xff, 0xff, 0xff, 0xff, 0xff}};
struct ether_addr 	saddr;					// Our ethernet address
struct ether_addr	gPPPoEEthernetAddress;	// Targets ethernet address
UInt16				gSessionID=0;
UInt8				gState=kPPPoE_Init_State;
#define	MAX_PPPOE_NAME_SIZE	256
UInt8				gACName[MAX_PPPOE_NAME_SIZE];
UInt8				gServiceName[MAX_PPPOE_NAME_SIZE];
pid_t				gProcessID, gProcessGroupID;
UInt32				gACNameLength=0,gServiceNameLength=0;
int					pppd_in_fd, pppd_out_fd;
boolean				bEscControlChar=TRUE;
boolean				bSyncPPP;
boolean				gbXmtTerminateAck=FALSE;
boolean				gbRcvTerminateAck=FALSE;
boolean				gbShutDown=FALSE;
int					gSyncOption;
unsigned 			gLastLcpId;
unsigned int		PPPRecvCount=0, PPPSendCount=0;
unsigned int 		PppdFrameErrors=0, PppdFrameBak2BakErrors=0;
unsigned int		gPADOCount=0;

#define	MAX_BAK_2_BAK_ERRORS	16

#define	kFoundACName		0x0001
#define kFoundServiceName	0x0002
#define kFoundHostUniq		0x0004
// Note, we always care about kFoundHostUniq
int					DontCareFoundMask = kFoundACName | kFoundServiceName;

char *sInterface;

#ifdef	LINUX2_2
typedef struct sockaddr_ll SOCKADDR;
#else
typedef struct sockaddr SOCKADDR;
#endif
SOCKADDR 	XcvSockAddress;

int			SockRecvSession;
int			SockRecvDiscovery;
int			SockSend;
#ifdef LINUX2_2
#define WRITE_DLI( a, b )\
(send(SockSend, (a), (b), 0))
#else
#define WRITE_DLI( a, b )\
(sendto(SockSend, (a), (b), 0, &XcvSockAddress, sizeof(XcvSockAddress)))
#endif

#define MAX_PPPFRAMEBUF		((1500+20)*2)
#define SESSION_DATA_OFFSET (ETH_H + sizeof(PPPoEPayloadHeader))

#define	kPPPoE_Discovery_eType		((UInt16)0x8863)
#define	kPPPoE_Session_eType		((UInt16)0x8864)

#define	FCS_INIT			((unsigned short)0xFFFF)
#define	FCS_TERM			((unsigned short)0xF0B8)

// CRC (aka FCS) lookup table from RFC 1662, pp. 19-20

static unsigned short		FCSTable[] = {
	0x0000, 0x1189, 0x2312, 0x329b, 0x4624, 0x57ad, 0x6536, 0x74bf,
	0x8c48, 0x9dc1, 0xaf5a, 0xbed3, 0xca6c, 0xdbe5, 0xe97e, 0xf8f7,
	0x1081, 0x0108, 0x3393, 0x221a, 0x56a5, 0x472c, 0x75b7, 0x643e,
	0x9cc9, 0x8d40, 0xbfdb, 0xae52, 0xdaed, 0xcb64, 0xf9ff, 0xe876,
	0x2102, 0x308b, 0x0210, 0x1399, 0x6726, 0x76af, 0x4434, 0x55bd,
	0xad4a, 0xbcc3, 0x8e58, 0x9fd1, 0xeb6e, 0xfae7, 0xc87c, 0xd9f5,
	0x3183, 0x200a, 0x1291, 0x0318, 0x77a7, 0x662e, 0x54b5, 0x453c,
	0xbdcb, 0xac42, 0x9ed9, 0x8f50, 0xfbef, 0xea66, 0xd8fd, 0xc974,
	0x4204, 0x538d, 0x6116, 0x709f, 0x0420, 0x15a9, 0x2732, 0x36bb,
	0xce4c, 0xdfc5, 0xed5e, 0xfcd7, 0x8868, 0x99e1, 0xab7a, 0xbaf3,
	0x5285, 0x430c, 0x7197, 0x601e, 0x14a1, 0x0528, 0x37b3, 0x263a,
	0xdecd, 0xcf44, 0xfddf, 0xec56, 0x98e9, 0x8960, 0xbbfb, 0xaa72,
	0x6306, 0x728f, 0x4014, 0x519d, 0x2522, 0x34ab, 0x0630, 0x17b9,
	0xef4e, 0xfec7, 0xcc5c, 0xddd5, 0xa96a, 0xb8e3, 0x8a78, 0x9bf1,
	0x7387, 0x620e, 0x5095, 0x411c, 0x35a3, 0x242a, 0x16b1, 0x0738,
	0xffcf, 0xee46, 0xdcdd, 0xcd54, 0xb9eb, 0xa862, 0x9af9, 0x8b70,
	0x8408, 0x9581, 0xa71a, 0xb693, 0xc22c, 0xd3a5, 0xe13e, 0xf0b7,
	0x0840, 0x19c9, 0x2b52, 0x3adb, 0x4e64, 0x5fed, 0x6d76, 0x7cff,
	0x9489, 0x8500, 0xb79b, 0xa612, 0xd2ad, 0xc324, 0xf1bf, 0xe036,
	0x18c1, 0x0948, 0x3bd3, 0x2a5a, 0x5ee5, 0x4f6c, 0x7df7, 0x6c7e,
	0xa50a, 0xb483, 0x8618, 0x9791, 0xe32e, 0xf2a7, 0xc03c, 0xd1b5,
	0x2942, 0x38cb, 0x0a50, 0x1bd9, 0x6f66, 0x7eef, 0x4c74, 0x5dfd,
	0xb58b, 0xa402, 0x9699, 0x8710, 0xf3af, 0xe226, 0xd0bd, 0xc134,
	0x39c3, 0x284a, 0x1ad1, 0x0b58, 0x7fe7, 0x6e6e, 0x5cf5, 0x4d7c,
	0xc60c, 0xd785, 0xe51e, 0xf497, 0x8028, 0x91a1, 0xa33a, 0xb2b3,
	0x4a44, 0x5bcd, 0x6956, 0x78df, 0x0c60, 0x1de9, 0x2f72, 0x3efb,
	0xd68d, 0xc704, 0xf59f, 0xe416, 0x90a9, 0x8120, 0xb3bb, 0xa232,
	0x5ac5, 0x4b4c, 0x79d7, 0x685e, 0x1ce1, 0x0d68, 0x3ff3, 0x2e7a,
	0xe70e, 0xf687, 0xc41c, 0xd595, 0xa12a, 0xb0a3, 0x8238, 0x93b1,
	0x6b46, 0x7acf, 0x4854, 0x59dd, 0x2d62, 0x3ceb, 0x0e70, 0x1ff9,
	0xf78f, 0xe606, 0xd49d, 0xc514, 0xb1ab, 0xa022, 0x92b9, 0x8330,
	0x7bc7, 0x6a4e, 0x58d5, 0x495c, 0x3de3, 0x2c6a, 0x1ef1, 0x0f78
};

#define CalculateFCS(oldFCS, theByte)	\
	((oldFCS >> 8) ^ FCSTable[(((unsigned char)oldFCS) ^ theByte) &\
 ((unsigned char)0xFF)])


/*
 ****************************************************************************
 * 
 * Begin Discovery handling routines
 *
 */

//
// Send PPPoE Active Discovery Initiation (PADI) packet
//
void
PPPoE_SendPADI( void )
{
  UInt16		packet_len;
  PPPoEPacket 	*packet;
  struct ethheader *pEthHeader;
  int numberOfBytesWritten;
  PPPoETag		*tag;
  
  pEthHeader = (struct ethheader *) gXmitFrame;

  pthread_mutex_lock( &gSendMutex );  

  SET_MAC_ADDR( pEthHeader->ether_dhost, broadcast.ether_addr_octet );
  SET_MAC_ADDR( pEthHeader->ether_shost, saddr.ether_addr_octet );  
  pEthHeader->ether_type = htons(kPPPoE_Discovery_eType);

  packet = (PPPoEPacket *)( &gXmitFrame[sizeof( struct ethheader )] );
  packet->payloadHeader.version 	= kPPPoE_Version;
  packet->payloadHeader.code 		= kPPPoE_Disc_PADI;
  packet->payloadHeader.SessionID 	= 0; 

  tag = &packet->tag;
  tag->type 	= htons(kPPPoE_Tag_ServiceName);
  tag->length 	= htons(gServiceNameLength);
  if( gServiceNameLength != 0 )
    {
	  memcpy( tag->data, gServiceName, gServiceNameLength );
	}
  NEXT_TAG_ITEM( tag );

  tag->type 	= htons(kPPPoE_Tag_Host_Uniq);
  tag->length 	= htons(sizeof(pid_t)+sizeof(saddr));
  memcpy( tag->data, &gProcessID, sizeof(pid_t) );
  memcpy( &tag->data[sizeof(pid_t)], saddr.ether_addr_octet, sizeof(saddr) );
  NEXT_TAG_ITEM( tag );

  packet_len = (UInt16)((char *)tag - (char *)packet);
  packet->payloadHeader.length = htons(packet_len - 6);
  gXmitFrameLength = ETH_H + packet_len;

  SET_STATE(kPPPoE_SentPADI_State);
  bSendRequest = TRUE;

  pthread_cond_signal( &gSendCond );
  pthread_mutex_unlock( &gSendMutex );  
}  //PPPoE_SendPADI

//
// Send PPPoE Active Discovery Request packet
//
static void	
PPPoE_SendPADR(	UInt8 *pACCookie, UInt16 ACCookieLength, 
				UInt8 *pRelayID,  UInt16 RelayIDLength )
{	
  UInt16		packet_len;
  PPPoETag		*tag;
  PPPoEPacket 	*packet;
  struct ethheader *pEthHeader;
  pEthHeader = (struct ethheader *) gXmitFrame;

  pthread_mutex_lock( &gSendMutex );  

// Not as many fields to fill in because PADI has already filled them in
  SET_MAC_ADDR(pEthHeader->ether_dhost, gPPPoEEthernetAddress.ether_addr_octet);
  packet = (PPPoEPacket *)&gXmitFrame[ETH_H]; 
  packet->payloadHeader.code		= kPPPoE_Disc_PADR;
  tag = &packet->tag;
  tag->type 	= htons(kPPPoE_Tag_ServiceName);
  tag->length 	= htons(gServiceNameLength);
  if( gServiceNameLength != 0 )
    {
	  memcpy( tag->data, gServiceName, gServiceNameLength );
	}
  NEXT_TAG_ITEM( tag );

  tag->type 	= htons(kPPPoE_Tag_Host_Uniq);
  tag->length 	= htons(sizeof(pid_t)+sizeof(saddr));
  memcpy( tag->data, &gProcessID, sizeof(pid_t) );
  memcpy( &tag->data[sizeof(pid_t)], saddr.ether_addr_octet, sizeof(saddr) );
  NEXT_TAG_ITEM( tag );

  if ( ACCookieLength )
    {
      tag->type   = htons(kPPPoE_Tag_AC_Cookie);
      tag->length = htons(ACCookieLength);
      memcpy( tag->data, pACCookie, ACCookieLength );
	  NEXT_TAG_ITEM( tag );
    }

  if ( RelayIDLength )
    {
      tag->type 	= htons(kPPPoE_Tag_Relay_Session_ID);
      tag->length 	= htons(RelayIDLength);
      memcpy( tag->data, pRelayID, RelayIDLength );
	  NEXT_TAG_ITEM( tag );
    }
  
  packet_len = (UInt16)((char *)tag - (char *)packet);
  packet->payloadHeader.length = htons(packet_len - 6);	//Adjust out payload header

  gXmitFrameLength = ETH_H + packet_len;

  SET_STATE(kPPPoE_SentPADR_State);
  bSendRequest = TRUE;

  pthread_cond_signal( &gSendCond );
  pthread_mutex_unlock( &gSendMutex );  
} // PPPoE_SendPADR

//
// Send PPPoE Active Discovery Termination (PADT) packet
// This must only be called when we are back at the main thread
//
void
PPPoE_SendPADT(void)
{
  UInt16		packet_len;
  PPPoEPacket 	*packet;
  struct ethheader *pEthHeaderPADS;
  struct ethheader *pEthHeader;
  int numberOfBytesWritten;
  PPPoETag		*tag;

  pEthHeader = (struct ethheader *) gXmitFrame;

  SET_MAC_ADDR(pEthHeader->ether_dhost, gPPPoEEthernetAddress.ether_addr_octet);
  SET_MAC_ADDR(pEthHeader->ether_shost, saddr.ether_addr_octet);  
  pEthHeader->ether_type = htons(kPPPoE_Discovery_eType);

  packet = (PPPoEPacket *)(&gXmitFrame[sizeof( struct ethheader )]);
  packet->payloadHeader.version 	= kPPPoE_Version;
  packet->payloadHeader.code 		= kPPPoE_Disc_PADT;
  packet->payloadHeader.SessionID 	= gSessionID;

  packet_len = (UInt16)((char *)&packet->tag - (char *)packet);
  packet->payloadHeader.length = htons(packet_len - 6);
  gXmitFrameLength = ETH_H + packet_len;
  WRITE_DLI(gXmitFrame, gXmitFrameLength);
}

//
// Gather any interesting information, TAGs.
// returns **tag pointing to next tag
// returns FoundWhat update with bits ored in for AC Name/Service Name if found
//
static	int	
PPPoE_HandleTag( PPPoETag **tag, int FoundWhat,
		UInt8 *pACCookie, UInt16 *pACCookieLength, 
		UInt8 *pRelayID,  UInt16 *pRelayIDLength,
		UInt8 **pFoundACName, UInt16 *FoundACNameLength )
{
  UInt16 Length = ntohs((*tag)->length);
  switch ( ntohs((*tag)->type) )
    {
    case kPPPoE_Tag_AC_Name:
	  if( gACNameLength == Length )
		{
		  if( !memcmp(gACName, (*tag)->data, Length) )
	    	FoundWhat |= kFoundACName;
		}
	  *pFoundACName = (*tag)->data;
	  *FoundACNameLength = Length;
      break;
    case kPPPoE_Tag_ServiceName:
	  if( gServiceNameLength == Length )
		{
		  if( !memcmp(gServiceName, (*tag)->data, Length) )
	    	FoundWhat |= kFoundServiceName;
		}
      break;
    case kPPPoE_Tag_AC_Cookie:
	  *pACCookieLength	= Length;
      if ( Length )
		memcpy( pACCookie, (*tag)->data, Length );
      break;
    case kPPPoE_Tag_Relay_Session_ID:
      *pRelayIDLength	= Length;
      if ( Length <= MAX_RELAY_ID_LENGTH )
		memcpy( pRelayID, (*tag)->data, Length );
	  else
	    *pRelayIDLength = 0;
      break;
	case kPPPoE_Tag_Host_Uniq:
	  if( (sizeof(pid_t)+sizeof(saddr)) == Length )
  		if( !memcmp( (*tag)->data, &gProcessID, sizeof(pid_t) ) &&
		    !memcmp( &(*tag)->data[sizeof(pid_t)],
		         saddr.ether_addr_octet, sizeof(saddr) ) )
		  FoundWhat |= kFoundHostUniq;		  
	  break;
    }
  
  NEXT_TAG_ITEM( *tag );
  return FoundWhat;
}

//
// Examines current tag if it is an error tag if prints it.
// Returns 1 if an error tag was found.
// When AvanceTagPointer is !=0. **tag will point to next tab.	
//
//#define LOG_PUTS( a, b ) syslog(LOG_ERROR, "\n"(a)"\n", (b));
#define LOG_PUTS( a, b ) syslog(LOG_ERR, (a))

static	boolean	
PPPoE_PrintErrorTag( PPPoETag **tag, int AdvanceTagPointer )
{
  boolean result = 0;
  // -16 bytes max for Relay Session ID Tag data, type & length, 
  // -6 bytes PADx header, 
  // -4 Error tag type&length, +1 null terminator
  UInt8 buff[1500-16-6-4+1];
  unsigned Length;

  Length = htons((*tag)->length);
  switch (htons((*tag)->type)) {
  case kPPPoE_Tag_Service_Name_Error:
    LOG_PUTS( "Service Name Error", stderr );
    result=TRUE;
    break;
  case kPPPoE_Tag_AC_System_Error:
    LOG_PUTS( "Access Concentrator System Error", stderr );
    result=TRUE;
    break;
  case kPPPoE_Tag_Generic_Error:
    LOG_PUTS( "Generic Error", stderr );
    result=TRUE;
    break;
  }
  if (result && Length) {
    if (Length < sizeof(buff)) {
  		memcpy(buff, (*tag)->data, Length);
		buff[Length] = '\0';
	  	syslog(LOG_ERR, "UTP-8 Data: %s", buff);
	}
  }

  if( AdvanceTagPointer )
	NEXT_TAG_ITEM( *tag );
  return( result );
}

//
// Scan PPPoE Packet for error tags and print them out
// returns 0 on no errors found !=0 when errors found
//
boolean
PPPoE_Scan4ErrorTag(PPPoETag *tag, char *pPayloadEnd)
{
  boolean foundOne = FALSE;

  while ((char *)tag + TAG_TYPE_LENGTH_SIZE <= pPayloadEnd)
  {
	if (PPPoE_PrintErrorTag(&tag, TRUE))
	  foundOne = TRUE;
  }
  return foundOne;
}

//
//
//
int
PPPoE_ProcessPADO(struct ethheader *pEthHeader, PPPoETag *tag, 
															char *pPayloadEnd)
{
  int		FoundWhat;
  // -16 bytes max for Relay Session ID Tag data, type & length, 
  // -6 bytes PADx header, 
  // -4 Cookie tag type&length
  UInt8		ACCookie[1500-16-6-4];
  UInt8		RelayID[MAX_RELAY_ID_LENGTH];
  UInt8 	*FoundACName=NULL;
  UInt16	ACCookieLength=0, RelayIDLength=0, FoundACNameLength=0;

  //Preset with the things we needn't look for
  FoundWhat = DontCareFoundMask;	

  while ((char *)tag + TAG_TYPE_LENGTH_SIZE <= pPayloadEnd)
  {
	  if (PPPoE_PrintErrorTag(&tag, FALSE))
		return(FALSE);
		
      FoundWhat = 
	  PPPoE_HandleTag(&tag, FoundWhat, 
  					  ACCookie, &ACCookieLength, RelayID, &RelayIDLength,
					  &FoundACName, &FoundACNameLength);
  }

  if (FoundWhat == (kFoundACName | kFoundServiceName | kFoundHostUniq)) {
      memcpy( 
  	  		(char *)&gPPPoEEthernetAddress, 
  			pEthHeader->ether_shost, 
  			sizeof( gPPPoEEthernetAddress ));
      PPPoE_SendPADR(ACCookie, ACCookieLength, RelayID, RelayIDLength);

	  if (FoundACName == NULL) {	//Should never be NULL
	  	return(FALSE);
	  }
	  // Note, we are and must be finished with the receive buffer before 
	  // we do tack on a null at the end of the ACName in the buffer.
	  FoundACName[FoundACNameLength] = '\0';
	  syslog(LOG_INFO, "Connecting to AC, \"%s\".", FoundACName);
	  return(TRUE);
  }else{
	  return(FALSE);
  }
}

//
//
//
void
PrintACNameAndServices(PPPoETag *p, char *pPayloadEnd)
{
  	FILE *pOut = stderr;
  	PPPoETag *tag;
  	UInt16 Length;
  
  	fprintf(pOut, "\nAccess Concentrator Name and Service Names:\n");

  	for (tag = p; (char *)tag < pPayloadEnd; NEXT_TAG_ITEM(tag))
  	{
  		if (kPPPoE_Tag_AC_Name == ntohs(tag->type)) {
  	    	Length = ntohs(tag->length);
			if (Length != 0) {
				fputs("  ", pOut);
			    fwrite(tag->data, 1, Length, pOut);
				fputc('\n', pOut);
				break;
			}
  		}
  	}

  	for (tag = p; (char *)tag < pPayloadEnd; NEXT_TAG_ITEM(tag))
  	{
  		if (kPPPoE_Tag_ServiceName == ntohs(tag->type))	{
  	    	Length = ntohs(tag->length);
			if (Length != 0) {
				fputs ("    ", pOut);
			    fwrite (tag->data, 1, Length, pOut);
				fputc ('\n', pOut);
			}
  		}
  	}
}

//
// Process kPPPoE_Discovery_eTypes from the ethernet
//
void	
PPPoE_ProcessPacket(char *pkt)
{
  struct ethheader *pEthHeader;
  PPPoEPacket *PPPoE_pkt = (PPPoEPacket *)(pkt+ETH_H);

  char *pPayloadEnd;
  UInt16 Length;
  PPPoETag *tag = &PPPoE_pkt->tag;

  pEthHeader = (struct ethheader *) pkt;
  pPayloadEnd =(char *)&PPPoE_pkt->tag + ntohs(PPPoE_pkt->payloadHeader.length);

  switch(PPPoE_pkt->payloadHeader.code)
    {
    case kPPPoE_Disc_PADO:	// Offer
	  gPADOCount++;
      if (gState == kPPPoE_SentPADI_State)
		  PPPoE_ProcessPADO(pEthHeader, tag, pPayloadEnd);
	  else if(gState == kPPPoe_GatherACNames)
		  PrintACNameAndServices(tag, pPayloadEnd);
      break;

    case kPPPoE_Disc_PADS:						// Session-confirmation
      if ( gState == kPPPoE_SentPADR_State )	// Did we send a request
		{
		  int FoundWhat=0;
		   
		  DEBUGOUT( "I see a PADS" );
		  if( !MAC_EQUAL(gPPPoEEthernetAddress.ether_addr_octet,
		                 pEthHeader->ether_shost) ) 
		    break;		//Don't talk to strangers
		  
		  /* Check for correct Session Name and Host-Uniq. */
  		  for( ;(char *)tag < pPayloadEnd; NEXT_TAG_ITEM( tag ) )
  		    {
		      Length = ntohs(tag->length);
		   	  if( kPPPoE_Tag_ServiceName == ntohs(tag->type) )
		        {
		   	      DEBUGOUT( "Found a ServiceName TAG" );
		   	      if( gServiceNameLength == Length )
		   	        if( !memcmp(gServiceName, tag->data, Length) )
	       	  	 	   FoundWhat |= kFoundServiceName;
		        }
		      else if( kPPPoE_Tag_Host_Uniq == ntohs(tag->type) )
		        {
	  	   	      if( (sizeof(pid_t)+sizeof(saddr)) == Length )
  		   	        if( !memcmp( tag->data, &gProcessID, sizeof(pid_t) ) &&
		   		        !memcmp( &tag->data[sizeof(pid_t)], 
		   		                 saddr.ether_addr_octet, sizeof(saddr) ) )
					  {
		   		      	FoundWhat |= kFoundHostUniq;
						DEBUGOUT( "Found Host-Uniq TAG" );
					  }
		     	}
 		    }

		  if( (FoundWhat & kFoundHostUniq) == 0 )
		    break;

		  tag = &PPPoE_pkt->tag;			
		  DEBUGOUT( "Scan4ErrorTag" );
		  DEBUGOUT2( "FoundWhat=0x%4.4X", FoundWhat );
	      if( PPPoE_Scan4ErrorTag( tag, pPayloadEnd ) )
		    { // error recovery - go back to the init state
			  // Note, we are accepting this error even if kFoundServiceName 
			  //   is not true.  kFoundHostUniq should be enough.
			  SET_STATE(kPPPoE_Init_State);
			  DEBUGOUT( "Error Tag found." );
  		  	  pthread_mutex_lock( &gSendMutex );  
  		  	  pthread_cond_signal( &gSendCond );
  		  	  pthread_mutex_unlock( &gSendMutex );  
		  	  DEBUGOUT( "Signal Retry Logic." );
			}
		  else if( FoundWhat == (kFoundServiceName | kFoundHostUniq) )
			{ // Alright we really got one
	      	  gSessionID	= PPPoE_pkt->payloadHeader.SessionID;
	      	  SET_STATE(kPPPoE_PPP_State);
			  DEBUGOUT( "Got SessionID." );
			  if (gServiceNameLength)
			  	syslog(LOG_INFO, "Service Name, %s.", gServiceName); 
	  	  	  syslog(LOG_INFO, "Connected SessionID, %d.", ntohs(gSessionID)); 
			} 	 
		    // If this isn't what we wanted, silently ignore it.
		}
      break;
	case kPPPoE_Disc_PADT:						// Session Terminated
      if (gState == kPPPoE_PPP_State)			// Do we have an active session
		{
		  if (!MAC_EQUAL(gPPPoEEthernetAddress.ether_addr_octet,
		                 pEthHeader->ether_shost)) 
		    break;		//Don't talk to strangers

		  if (gSessionID != PPPoE_pkt->payloadHeader.SessionID)
		    break;		  
  	  	  syslog(LOG_INFO, 
		  				"Received PADT for SessionID, %d.", ntohs(gSessionID)); 
	  	  gSessionID = 0;
		  // There are no Error tags defined to be used with PADT
		  // Place this here just in case somebody tries to.
	      PPPoE_Scan4ErrorTag( tag, pPayloadEnd );
	  	  SET_STATE(kPPPoE_LastState);	// kPPPoE_ReceivedPADT_State; //
		  kill(gProcessID, SIGHUP);		//Wake up all of those selects
		}
	  break;
    default:
      DEBUGOUT2("PPPoE_ProcessPacket should not have gotten this value 0x%2.2x", 
		  PPPoE_pkt->payloadHeader.code );
      break;
    }
}

//
//
//
boolean PPPoE_Discovery( void )
{
  PPPoEPacket 	*packet;
  struct ethheader *pEthHeader;
  int 			numberOfBytesWritten, Retcode, PADIRestarts;
  PPPoETag *tag;
  unsigned		ReXmitDelay;
  struct timeval CurrentTime;
  struct timezone TimeZone;
  struct timespec Timeout;

  // This has to do with how many times we got beyond kPPPoE_SentPADI_State
  // and failed to establish a session.
  PADIRestarts = 0;
  TimeZone.tz_minuteswest = 0;
  TimeZone.tz_dsttime = 0;

  while(gState == kPPPoE_Init_State)
  {
	if (PADIRestarts++ > 3) { 
		// 3 strikes your out.
	    SET_STATE(kPPPoE_LastState);
	    break;
	}
    gPADOCount=0;
  	PPPoE_SendPADI();
	bSendRequest = FALSE;
	// Else our retransmit logic will use 1 sec. interval twice

    syslog(LOG_INFO, "Sending PADI.");
  	
	//Transmit and Retransmit until we succeed or timelimit is reached
  	ReXmitDelay = 1;			// Start at 1 sec and double on retries
    pthread_mutex_lock(&gSendMutex);
  	do 
	{
	    // All Discovery frames are sent from this loop
	    // They are built elsewhere and we are signaled to send them
  	    numberOfBytesWritten = 		
  	    WRITE_DLI(gXmitFrame, gXmitFrameLength);

	    if (gState == kPPPoE_SentPADT_State) {
	    	  // We just sent a session terminate. Now we must move out of the
	    	  // retry loop and start over with PADI.
	    	  SET_STATE(kPPPoE_Init_State);
	    	  break;
	    }
  	    gettimeofday(&CurrentTime, &TimeZone);
  	    Timeout.tv_sec 	= CurrentTime.tv_sec + ReXmitDelay;
  	    Timeout.tv_nsec = CurrentTime.tv_usec * 1000; 
  	    Retcode = pthread_cond_timedwait(&gSendCond, &gSendMutex, &Timeout);

	    if (gState == kPPPoE_PPP_State)
	    	break;

	    if (bSendRequest) {	
	    	// State change send new frame
	    	ReXmitDelay = 1;
	    	bSendRequest = FALSE;
	    	Retcode = ETIMEDOUT;
        	syslog( LOG_INFO, "Sending PADR." );
	    } else {
  	    	ReXmitDelay *= 2;
	    	if (ReXmitDelay > 8) { 
	    	    if (gState == kPPPoE_SentPADR_State) {
	    		    SET_STATE(kPPPoE_Init_State); //Try again but start over
				} else {
					SET_STATE(kPPPoE_LastState);	// Give up
				    if (gPADOCount)
					   	syslog(LOG_INFO, 
							   "Received %u non-matching PADO responses.",
							   gPADOCount);
					else
					   	syslog(LOG_INFO, "No PADO responses received.");
				}
	    		break;
	    	}
	    	if (Retcode == ETIMEDOUT) {
	    	    if (gState == kPPPoE_SentPADR_State) {
	    		    syslog(LOG_INFO, "Re-sending PADR.");
	    		} else {
				    if (gPADOCount)
					   	syslog(LOG_INFO, 
							   "Received %u non-matching PADO responses.",
							   gPADOCount);
					else
					   	syslog(LOG_INFO, "No PADO responses received.");
	    		    syslog(LOG_INFO, "Re-sending PADI.");
					gPADOCount=0;
				}
			}
	    }
	    if (gState == kPPPoE_LastState) 
	    	break;
  	}while (Retcode == ETIMEDOUT);
  	pthread_mutex_unlock(&gSendMutex);  
  }
  return(gState == kPPPoE_PPP_State);
}
/*
 ****************************************************************************
 * 
 * Begin Session handling routines
 *
 */

//
// Process ethernet data. Output as PPP data to pppd.
//
void
PPPoE_ProcessAsyncPPP( char *pkt )
{
  PPPoEPacket *PPPoE_pkt = (PPPoEPacket *)&pkt[ETH_H];

  UInt8	 the_PPP_Packet[MAX_PPPFRAMEBUF];// 20 bytes paranoid padding
  unsigned	short	fcs = FCS_INIT;
  unsigned  Length;
  UInt8	fcsByte, EthByte;
  UInt8 *pPPPData, *pEthByte, *pEndData, *pNextXfer;
  int SegmentLength;
  int BytesWriten;
  int TotalBytes;
 
  const char	startSequence[4] = { kDLC_FLAG, 0xff, kESC, 0x23 };

  if( gState != kPPPoE_PPP_State )
    return;

  // Note we stored SessionID in network form ie. no need to ntohs/htons.
  if( PPPoE_pkt->payloadHeader.SessionID != gSessionID )
    return;

  pPPPData = the_PPP_Packet;
  Length = (unsigned) ntohs(PPPoE_pkt->payloadHeader.length);
  pEthByte = (UInt8 *)(&PPPoE_pkt->tag);
  pEndData = pEthByte + Length;

#ifdef RECEIVE_ALL
  // This does not work for all concentrators.  Some concentrators reject the 
  // the attempt to set ACCM to 0x0.  Thus, pppd expects to see all control
  // characters escaped.  This can be worked around by setting receive-all
  // option on pppd.  pppd will then accept non-escaped control characters
 
  // Look for PAP or CHAP SUCCESS, after that we will assume LCP negotiation is 
  // complete.  And we will stop escaping control characters ever after.
   if( bEscControlChar )
	if((pEthByte[0] == 0x0C0 && pEthByte[1] == 0x023 && pEthByte[2] == 0x002)||
	   (pEthByte[0] == 0x0C2 && pEthByte[1] == 0x023 && pEthByte[2] == 0x003)  )
      bEscControlChar = FALSE;
#endif

  fcs = CalculateFCS( fcs, 0xff );
  fcs = CalculateFCS( fcs, 0x03 );
  // Note fcs is computed before the data is escaped
  
  memcpy( pPPPData, startSequence, 4 );	// Includes beginning PPP_FLAG
  pPPPData += 4;

  if( bEscControlChar )
    for ( pNextXfer=pEthByte; pEthByte < pEndData; pEthByte++ )
      {
    	EthByte = *pEthByte;
        fcs = CalculateFCS( fcs, EthByte );
        if( EthByte <  0x20		 ||
    	    EthByte == kDLC_FLAG || 	// 0x7E
    		EthByte == kESC          )	// 0x7D
    	  {
    		if( pNextXfer != pEthByte )
    		  {
    		    SegmentLength = pEthByte-pNextXfer;
    	    	memcpy( pPPPData, pNextXfer, SegmentLength );
    			pPPPData += SegmentLength;
    		  }
    		*pPPPData++ = kESC;
    		*pPPPData++ = EthByte ^ 0x20;
    	    pNextXfer   = pEthByte +1;
    	  }
      }
  else
    for ( pNextXfer=pEthByte; pEthByte < pEndData; pEthByte++ )
      {
    	EthByte = *pEthByte;
        fcs = CalculateFCS( fcs, EthByte );
        if( 
    	    EthByte == kDLC_FLAG || 	// 0x7E
    		EthByte == kESC          )	// 0x7D
    	  {
    		if( pNextXfer != pEthByte )
    		  {
    		    SegmentLength = pEthByte-pNextXfer;
    	    	memcpy( pPPPData, pNextXfer, SegmentLength );
    			pPPPData += SegmentLength;
    		  }
    		*pPPPData++ = kESC;
    		*pPPPData++ = EthByte ^ 0x20;
    	    pNextXfer   = pEthByte +1;
    	  }
      }

  if( pNextXfer != pEthByte )
	{
	  SegmentLength = pEthByte-pNextXfer;
	  memcpy( pPPPData, pNextXfer, SegmentLength );
	  pPPPData += SegmentLength;
	}

  fcs ^= 0xffff;

  fcsByte = fcs & 0xff;
  if( (bEscControlChar && (fcsByte <  0x20))    ||
      					   fcsByte == kDLC_FLAG ||
  	  					   fcsByte == kESC         )
    {
	  *pPPPData++ = kESC;
	  *pPPPData++ = fcsByte ^ 0x20;
	}
  else
	  *pPPPData++ = fcsByte;

  fcsByte = fcs >> 8;
  if( (bEscControlChar && (fcsByte <  0x20))    ||
      					   fcsByte == kDLC_FLAG ||
  	  					   fcsByte == kESC         )
    {
	  *pPPPData++ = kESC;
	  *pPPPData++ = fcsByte ^ 0x20;
	}
  else
	  *pPPPData++ = fcsByte;
    
  *pPPPData++ =  kDLC_FLAG;

  TotalBytes = pPPPData - the_PPP_Packet;
  BytesWriten = write( pppd_out_fd, the_PPP_Packet, TotalBytes );
  if(BytesWriten == TotalBytes) {
  	  PPPRecvCount++;
	  return;
  } else if (BytesWriten == 0) {
	  syslog(LOG_INFO, "PPPoE_ProcessAsyncPPP: Zero bytes writen to pty device.");
  } else if (BytesWriten < 0) {
      if (gState != kPPPoE_PPP_State) {
	  	  return;
	  }else if (errno == EINTR) {
          BytesWriten = write( pppd_out_fd, the_PPP_Packet, TotalBytes );
  	      if (BytesWriten != TotalBytes)
	  	  	  syslog( LOG_INFO, "PPPoE_ProcessAsyncPPP: Not all data was"
					" writen to pty device %d!=%d", BytesWriten, TotalBytes );
		  else
		   	  PPPRecvCount++;
      } else {
	      syslog( LOG_INFO, "FAILED(%d): write(pppd_out_fd,,), %s(%d).",
										__LINE__, strerror(errno), errno ); 
	  }
  } else {
	  syslog( LOG_INFO, "PPPoE_ProcessAsyncPPP: Not all data was writen to "
	  			"pty device %d!=%d", BytesWriten, TotalBytes );
  }
  return;
}

//
// Process ethernet data. Output as Sync PPP data to pppd.
//
void
PPPoE_ProcessSyncPPP( char *pkt )
{
  PPPoESession *PPPoE_pkt = (PPPoESession *)&pkt[ETH_H];

  UInt8 *pPPPFrame;
  int BytesWriten;
  int TotalBytes;
 
  if (gState != kPPPoE_PPP_State)
      return;

  // Note we stored SessionID in network form ie. no need to ntohs/htons.
  if (PPPoE_pkt->payloadHeader.SessionID != gSessionID)
      return;

  // The PPP kernel driver will accept a frame with or without the
  // DLC Address/Control field eg. 0xFF 0x03.  So we will leave it off for now.
  pPPPFrame = PPPoE_pkt->data;
  TotalBytes = (unsigned)ntohs(PPPoE_pkt->payloadHeader.length);

  BytesWriten = write(pppd_out_fd, pPPPFrame, TotalBytes);
  if(BytesWriten == TotalBytes) {
  	  PPPRecvCount++;
	  return;
  } else if (BytesWriten == 0)	{
	  syslog(LOG_INFO, "PPPoE_ProcessSyncPPP: Zero bytes writen to pty device.");
  } else if (BytesWriten < 0)	{
	  if (gState != kPPPoE_PPP_State) {
	  	  return;
	  } else if (errno == EINTR) {
      	  BytesWriten = write( pppd_out_fd, pPPPFrame, TotalBytes );
  	      if (BytesWriten != TotalBytes)
	  	      syslog( LOG_INFO, "PPPoE_ProcessSyncPPP: Not all data was"
					" writen to pty device %d!=%d", BytesWriten, TotalBytes );
		  else
  	  	    	PPPRecvCount++;
      } else {
  	    	syslog(LOG_INFO, "FAILED(%d): write(pppd_out_fd,,), %s(%d).", 
  	  									__LINE__, strerror(errno), errno);
	  }
  } else {
	  syslog(LOG_INFO, "PPPoE_ProcessSyncPPP: Not all data was writen to"
	  							" pty device %d!=%d", BytesWriten, TotalBytes);
  }
  return;

}

//
// Receive thread for incoming packets
// Evaluates PPPoE Receive Packets from the network interface
// Call appropriate function to process packet, as needed.
//
//
void *
DoReceivePacketThread( void *notused )
{
  	int Length, ReadyCount, read_socket, maxfd1;
  	fd_set Recv_fd_set;
  	u_char packet[1518];
  	struct ethheader *pEthHeader = (struct ethheader *)packet;
  	PPPoEPayloadHeader *PPPoE_hdr = (PPPoEPayloadHeader *)&packet[ETH_H];
    struct timeval CurrentTime;
    struct timeval Timeout;
    struct timeval Timer;
    struct timezone TimeZone;

  	syslog(LOG_INFO, "Related PIDs: %d %d %d", gProcessGroupID, 
						gProcessID, getpid());

    TimeZone.tz_minuteswest = 0;
    TimeZone.tz_dsttime = 0;

  	FD_ZERO(&Recv_fd_set);	
  	maxfd1 = MAX(SockRecvSession, SockRecvDiscovery);
 	maxfd1++;

  	while(gState != kPPPoE_LastState)
  	{
  		FD_SET( SockRecvSession, &Recv_fd_set );
  		FD_SET( SockRecvDiscovery, &Recv_fd_set );
  	    Timer.tv_sec = (INPUT_TIMEOUT_DELAY/4) ? (INPUT_TIMEOUT_DELAY/4) : 1;
  	    Timer.tv_usec = 0;
  	  	ReadyCount = select(maxfd1, &Recv_fd_set, NULL, NULL, &Timer);
		if (ReadyCount <= 0) {
			if (ReadyCount == 0) 
				continue;
			if (errno == EINTR)
	  			continue;
			syslog(LOG_INFO, "FAILED(%d): select(), %s(%d).", __LINE__, 
													strerror(errno), errno); 
			goto ExitNow;
		}
		for (; ReadyCount > 0; ReadyCount--) 
		{
		  	if (FD_ISSET(SockRecvSession, &Recv_fd_set)) {
		  		read_socket = SockRecvSession;
				FD_CLR(SockRecvSession, &Recv_fd_set);
		  	}else if(FD_ISSET(SockRecvDiscovery, &Recv_fd_set)) {
		  		read_socket = SockRecvDiscovery;
				FD_CLR(SockRecvDiscovery, &Recv_fd_set);
		  	}

  		  	Length = 
			recvfrom( read_socket, packet, sizeof(packet), 0, NULL, NULL );
		  	if (Length > 0) {
		  	  	// Ignore packet, if it is not addressed to us.  
		  	  	// This will also ignore broadcast packets which is good.
  		  	  	if ( !MAC_EQUAL( pEthHeader->ether_dhost, (char *)&saddr ) )
  		  	  	  	continue;
  
  		  	  	if (Length < 					// Consistency check
  		  	  		ETH_H+sizeof(PPPoEPayloadHeader)+ntohs(PPPoE_hdr->length)) {
		  			syslog(LOG_INFO, 
		  				"STRANGE: Short recv packet, %d bytes, expected %d.",
		  				Length, 
		  				ETH_H+
						   sizeof(PPPoEPayloadHeader)+ntohs(PPPoE_hdr->length));
  		  	  	  	continue;
		  	  	}

  		  	  	if (ntohs(pEthHeader->ether_type) == kPPPoE_Session_eType) {
  					PPPoESession *PPPoE_pkt = (PPPoESession *)&packet[ETH_H];
  					UInt8 *pPPPFrame = PPPoE_pkt->data;

					if( bSyncPPP )
  	  					PPPoE_ProcessSyncPPP((u_char *)packet);
					else
  	  					PPPoE_ProcessAsyncPPP((u_char *)packet);

  					if (pPPPFrame[0] == 0x0C0 &&		// Protocol,
  					 	pPPPFrame[1] == 0x021 &&		// LCP
  					 	pPPPFrame[2] == 0x006 &&		// Code, Terminate-Ack
  					 	pPPPFrame[3] == gLastLcpId) {   // ID
							gbRcvTerminateAck = TRUE;			
							syslog(LOG_INFO, "Received LCP Terminate-Ack.");
							SET_STATE(kPPPoE_LastState);
  					}
  		  	  	} else if (ntohs(pEthHeader->ether_type) == 
													kPPPoE_Discovery_eType) {
  		  	  	  	PPPoE_ProcessPacket(packet);
				}
  		  	}else if (Length < 0) {
				if (errno == EINTR) 
					continue;
		  		syslog(LOG_INFO, "FAILED(%d): recvfrom(), %s(%d).", 
											__LINE__, strerror(errno), errno); 
	  			goto ExitNow;
		  	}
		}
  	}
ExitNow:
	SET_STATE(kPPPoE_LastState);
	kill(gProcessID, SIGHUP);	//Wake up everybody!
  	pthread_exit(0);
}

//
// Reads from pppd with timeout based on INPUT_TIMEOUT_INTERVAL and 
// INPUT_TIMEOUT_COUNT
//
// Returns: Number of bytes read into buffer FrameSize.  If value returned is
// negative errno contains the reason given by one of the library functions
// called on in implimenting this functions.  Contents of buffer, pFrame, 
// is undefined.  errno is set to zero on return when timeout or when not in 
// PPP state.
//
int
ReadPppd(unsigned char *pFrame, int FrameSize)
{
	fd_set pppd_in_fd_set;
	struct timeval Timer;
	int ReadyCount, BytesRead, Error, TimeoutCount;

	TimeoutCount = 0;
	while(gState == kPPPoE_PPP_State) 
	{
		BytesRead = 0;
		FD_ZERO(&pppd_in_fd_set);	
  		FD_SET(pppd_in_fd, &pppd_in_fd_set);
   		Timer.tv_sec = INPUT_TIMEOUT_INTERVAL;
  		Timer.tv_usec = 0;
  		ReadyCount = select(pppd_in_fd+1, &pppd_in_fd_set, NULL, NULL, &Timer);
  		if (ReadyCount == 0) {
			// Actual time may be greater that what is represented by
			// TimeoutCount, because an EINTR causes a given interval to be 
			// restarted.  This is not a problem, if we are going to timeout, 
			// we will timeout.  Extra cycles for precision in this area, IMHO,
			// is not warranted. We just need a rough limit on how long to wait.
			TimeoutCount++;
			if (TimeoutCount >= INPUT_TIMEOUT_COUNT) {
  				syslog(LOG_INFO, 
			    	"Timeout, waiting for data from pppd, %d.", __LINE__);
				errno = 0;
				return -1;
			}
			continue;
  		}
  		if (ReadyCount < 0) {
			if (gState != kPPPoE_PPP_State || errno == EINTR)
				continue;
			if (errno == EIO)
				return ReadyCount;
			syslog(LOG_INFO, "FAILED(%d): select(,,), %s(%d).",
											__LINE__, strerror(errno), errno);
			return ReadyCount;
  		} 
		Error = ioctl(pppd_in_fd, FIONREAD, &BytesRead);
		if (Error < 0) {
			if (gState != kPPPoE_PPP_State || errno == EINTR)
				continue;
  			syslog(LOG_INFO, "FAILED(%d): ioctl(,,), %s(%d).", 
											__LINE__, strerror(errno), errno);
			return Error;
		}
		if (BytesRead == 0)
			return 0;				// Assume EOF on input device
		if (BytesRead > FrameSize) 
			BytesRead = FrameSize;
  		BytesRead = read(pppd_in_fd, pFrame, BytesRead);
  		if (BytesRead < 0) {
			if (gState != kPPPoE_PPP_State || errno == EINTR || errno == EAGAIN)
				continue;
			if (errno != EIO)
				syslog(LOG_INFO, "FAILED(%d): read(,,), %s(%d).", 
									__LINE__, strerror(errno), errno );
		}
		return BytesRead;
	} //while(gState != kPPPoE_PPP_State) 
	errno = 0;
	return -1;
}	
//
// Auto detect framing method - Sync vs. Async 
//
// Returns: FRAME_METHOD_SYNC, FRAME_METHOD_ASYNC, or -1 on failure or 
// when not in PPP state (errno is undefined).  On a success *pBytesRead 
// will contains a count of useable bytes in buffer pFrame.
//
int
AutoDetectFrameMethod(unsigned char *pB, int BufferSize, int *pBytesRead)
{
	int i, j, BytesRead;

	*pBytesRead = 0;
	for (i=0; i<5; i++) 
  	{
	  	if (gState != kPPPoE_PPP_State)
			return -1;
		  
  	    BytesRead = 
  		ReadPppd(pB, BufferSize);
	  	if (gState != kPPPoE_PPP_State)	
			return -1;
  	    if (BytesRead <= 0)
			return -1;

  	    // Here we are trying to auto sense if the pppd has sync selected.
		// We have read at least one byte.
  	    if (pB[0] == kDLC_FLAG) {
			for(j=1; j<BytesRead && pB[j]==kDLC_FLAG; j++);
			if (j+5 > BytesRead) continue;
			if (pB[j+0] == 0x0FF &&
				pB[j+1] == kESC &&
				pB[j+2] == (0x003 ^ 0x020) &&
				pB[j+3] == 0x0C0 &&
				pB[j+4] == 0x021) {
				*pBytesRead = BytesRead;
			    bSyncPPP = FALSE;
  	    	    return FRAME_METHOD_ASYNC;
			}
			continue;
		}  	 	  	
  		if (BytesRead < MIN_SYNC_PPP_FRAME)
			continue;
  	    if (pB[0] == 0x0FF &&
			pB[1] == 0x003 &&
			pB[2] == 0x0C0 &&		// Must look like LCP
			pB[3] == 0x021) {
			*pBytesRead = BytesRead;
		    bSyncPPP = TRUE;
 			return FRAME_METHOD_SYNC;
		}
	}
  	syslog(LOG_INFO, "Could not auto-detect framing method Sync/Async.");
  	syslog(LOG_INFO, "Try adding sync or async to the pppoe command line.");
	return -1;
}

//
// Handle ASYNC formated PPP data.
// Loop till state is no longer PPP or some form of unacceptable error.
// e.g. EOF, EOI, etc
//
void
AsyncPppdLoop(unsigned char *pFrame, int FrameSize, int BytesRead)
{
	int NextByte, SessionData, EndOfRead, NumBytesToRead;
	boolean bSpecial, bPreviousDlcFlag, bDoOnce0xFF03, bEsc;
	unsigned Length;
	PPPoEPacket	*packet;
	packet = (PPPoEPacket *)&pFrame[ETH_H]; 

	syslog(LOG_INFO, "Operating in Async mode.");
	bPreviousDlcFlag =  FALSE;
	bSpecial = bEsc = FALSE;
	bDoOnce0xFF03 = TRUE;
	SessionData = SESSION_DATA_OFFSET;

	do{	// while ( gState == kPPPoE_PPP_State )
	  	NextByte = SessionData;
	    EndOfRead = SessionData + BytesRead;
		for(; NextByte < EndOfRead; NextByte++)
	  	{
	  	    if (bDoOnce0xFF03)
			    if (SessionData == (SESSION_DATA_OFFSET+2))
	  	  	    	if (pFrame[SESSION_DATA_OFFSET]   == 0xFF &&
	  	    	        pFrame[SESSION_DATA_OFFSET+1] == 0x03) {
				  	    bDoOnce0xFF03 = FALSE;
					  	SessionData = SESSION_DATA_OFFSET;
				  	}

			if (bSpecial) { 
				if (bEsc) {
					pFrame[SessionData++] = pFrame[NextByte] ^ 0x20;
			  		bSpecial = bEsc = FALSE;
					continue;
				}
	  			if (bPreviousDlcFlag) {
					if (pFrame[NextByte] == kDLC_FLAG)
						continue;

					bSpecial = bPreviousDlcFlag = FALSE;
				}
				// If none of the above is true we just fall through
				// and handle as normal data
			} //if( bSpecial )

	 		if (pFrame[NextByte] == kESC) {
				if (++NextByte < EndOfRead) {
					// Handle ESC inline.  This will avoid another 
					// pass through the loop.
					pFrame[SessionData++] = pFrame[NextByte] ^ 0x20;
					continue;
				} else {
				  	// We don't have to fixup ++NextByte beause this loop
					// is going to exit, and NextByte get's re-inited later.
	 				bSpecial = bEsc = TRUE;
	 				continue;
				}
	 		}

			if (pFrame[NextByte] == kDLC_FLAG) {
				if (SessionData != SESSION_DATA_OFFSET) {
					Length = SessionData - SESSION_DATA_OFFSET -2;
				    packet->payloadHeader.length = htons(Length);
				    WRITE_DLI(pFrame, SESSION_DATA_OFFSET + Length);
					PPPSendCount++;
	  				// Monitor for LCP Terminate-Ack
	  				if (pFrame[SESSION_DATA_OFFSET+0] == 0x0C0 &&	// Protocol
	  					pFrame[SESSION_DATA_OFFSET+1] == 0x021) {	// LCP
					  	if (pFrame[SESSION_DATA_OFFSET+2] == 0x006) {
							// Code, Terminate-Ack 
					  	  	// We are sending a LCP Terminate-Ack, all is over 
							gbXmtTerminateAck = TRUE;			
				  			syslog(LOG_INFO, "Sent LCP Terminate-Ack.");
					  	  	SET_STATE(kPPPoE_LastState);
					  	} else  
	  				  		gLastLcpId = pFrame[SESSION_DATA_OFFSET+3];// ID
	  				}				  
	 			}
	 			bSpecial = bPreviousDlcFlag = bDoOnce0xFF03 = TRUE;
	 			bEsc = FALSE;
	 			SessionData = SESSION_DATA_OFFSET;
	 		  	continue;
	 		}
			pFrame[SessionData++] = pFrame[NextByte];
	  	} //for( ; NextByte < NextRead; NextByte++ )

		BytesRead = ReadPppd(&pFrame[SessionData], FrameSize-SessionData);
		if (BytesRead <= 0)
	    	return;

	}while(gState == kPPPoE_PPP_State);
	return;
}
//
// Handle SYNC formated PPP data.
// Loop till state is no longer PPP or some form of unacceptable error.
// e.g. EOF, EOI, etc
//
void
SyncPppdLoop(unsigned char *pFrame, int FrameSize, int BytesRead)
{
	/*
	SYNC data against pty port notes.  
	1) PPP Kernel Driver expects sync ports to not accept any data at all
	   if there isn't room for the entire frame.  A pseudo tty (pty) will accept
	   a partial write this would cause some lost data when used with pppd's 
	   sync option.  To fix this we must assert the HDLC line discipline on 
	   the master side of the pty.
	2) The N_HDLC line discipline must be loaded before data from pppd will be
	   guranteed to have the correct possitioning in the buffer.
	3) With the N_HDLC line discipline, we no longer get EIO errors on read, 
	   have to watch for LCP Terminate-ACK in both directions.

	hdlc line discipline on a pty port acts a little funny.
	When pppd closes the slave side there is no immediate indicator
	on the master side.  So we now have a Timer running.
	When the select times out, the count is 1 instead of 0, and if we read 
    from the port we block.  So we now check to see how much there is to read.
    If it's zero assume we treat it as an EOF and exit.
	Note, if we did a 2ND call to select it would complete immediately.  
	*/

	unsigned Length;
	int hdlc_disc = N_HDLC;					// HDLC discipline
	PPPoEPacket	*packet;
	packet = (PPPoEPacket *)&pFrame[ETH_H]; 

	// Handle SYNC formated PPP data
    syslog(LOG_INFO, "Operating in Sync mode.");

	// Set the current tty to the hdlc discipline
	if (ioctl(pppd_in_fd, TIOCSETD, &hdlc_disc) < 0) {
		syslog(LOG_INFO, "Required N_HDLC line discipline for sync option is "
	  										"not supported by the system." );
		return;
      	/* 
	  	Reminder - I had to add a line to /etc/conf.modules so modprobe would
      	auto load the driver module n_hdlc.o.
	  	e.g. alias tty-ldisc-13 n_hdlc
		*/
	}
	if (BytesRead >= 3)	//Adjust possition for sync routine
		memmove(&pFrame[SESSION_DATA_OFFSET],
				&pFrame[SESSION_DATA_OFFSET+2], BytesRead-2);
	else
		BytesRead = 0;

	do{ // while ( gState == kPPPoE_PPP_State )
		if (BytesRead >= MIN_SYNC_PPP_FRAME) {
			PppdFrameBak2BakErrors=0;	
			Length = BytesRead-2;		// Remove the 0xFF03
			packet->payloadHeader.length = htons(Length);

			WRITE_DLI(pFrame, SESSION_DATA_OFFSET + Length);
			PPPSendCount++;
			// Monitor for LCP Terminate-Ack
			if (pFrame[SESSION_DATA_OFFSET+0] == 0x0C0 &&	// Protocol
				pFrame[SESSION_DATA_OFFSET+1] == 0x021) {	// LCP
				if (pFrame[SESSION_DATA_OFFSET+2] == 0x006) {
					// Code, Terminate-Ack 
					// We are sending a LCP Terminate-Ack, all is over 
					gbXmtTerminateAck = TRUE;			
					syslog(LOG_INFO, "Sent LCP Terminate-Ack.");
			  	  	SET_STATE(kPPPoE_LastState);
					return;
				}else  
					gLastLcpId = pFrame[SESSION_DATA_OFFSET+3];// ID
			}
		}
		// Make it easy to toss the address and control field by allowing the
		// Read data to start loading at the PPPoE Length location.
		BytesRead = 
		ReadPppd(&pFrame[SESSION_DATA_OFFSET-2], 
										FrameSize-(SESSION_DATA_OFFSET-2));
		if (BytesRead < MIN_SYNC_PPP_FRAME || 
			packet->payloadHeader.length != htons(0xFF03)) {
			if (BytesRead <= 0)
				return;
			PppdFrameErrors++;
			PppdFrameBak2BakErrors++;
			if (PppdFrameBak2BakErrors < MAX_BAK_2_BAK_ERRORS) {
				syslog(LOG_DEBUG, "Bad Sync PPP frame, 0x%4.4X.", 
				  						ntohs(packet->payloadHeader.length));
				BytesRead = 0;
				continue;
			} else {
				syslog(LOG_INFO,
						"Too many Back to Back frame errors from pppd.", 
			  			ntohs(packet->payloadHeader.length));
				return;
			}
		}
   	}while(gState == kPPPoE_PPP_State);
}
//
// Determine Framing method and read packets from pppd and send out ethernet
// as PPPoE frames.
//
void
Pppd2Ethernet(void)
{
	UInt8 the_Frame[ETH_H+MAX_PPPFRAMEBUF];	// 20 bytes paranoid padding
	int BytesRead;

	struct ethheader *pEthHeader;
	PPPoEPacket	*packet;

	pEthHeader = (struct ethheader *) the_Frame;
	// Construct Ethernet header once
	SET_MAC_ADDR(pEthHeader->ether_dhost, gPPPoEEthernetAddress.ether_addr_octet);
	SET_MAC_ADDR(pEthHeader->ether_shost, saddr.ether_addr_octet);  
	pEthHeader->ether_type = htons(kPPPoE_Session_eType);

	// Construct PPPoE payload header once
	packet = (PPPoEPacket *)&the_Frame[ETH_H]; 
	packet->payloadHeader.version	= kPPPoE_Version;
	packet->payloadHeader.code		= 0;
	packet->payloadHeader.SessionID	= gSessionID;

	gLastLcpId = (unsigned)(-1);

	BytesRead = 0;
  	if (gSyncOption == FRAME_METHOD_AUTO) {
		gSyncOption = 
		AutoDetectFrameMethod(&the_Frame[SESSION_DATA_OFFSET],
  	  						sizeof(the_Frame)-SESSION_DATA_OFFSET, &BytesRead); 
	}
	if (gSyncOption == FRAME_METHOD_SYNC) {
		SyncPppdLoop(the_Frame, sizeof(the_Frame), BytesRead);
	} else if (gSyncOption == FRAME_METHOD_ASYNC) {
		AsyncPppdLoop(the_Frame, sizeof(the_Frame), BytesRead);
	}
	SET_STATE(kPPPoE_LastState);
	return;
}

/*
 ****************************************************************************
 * 
 * Begin misc routines
 *
 */
//
// Setup Receive and Send sockets for ethernet interface
//
int 
InitSockets( void )
{
  int Result;
  struct ifreq ifr;

  // Create socket to receive session packets from interesting interface.
  SockRecvSession = 
#ifdef LINUX2_2
  socket(PF_PACKET, SOCK_RAW, htons(kPPPoE_Session_eType));
#else
  socket(AF_INET, SOCK_PACKET, htons(kPPPoE_Session_eType));
#endif
  if (SockRecvSession < 0) {
	syslog(LOG_INFO, "FAILED(%d): socket(,, 0x%4.4X), %s(%d).", 
					__LINE__, kPPPoE_Session_eType, strerror(errno), errno); 
    return(SockRecvSession);
  }
#ifdef LINUX2_2
  memset(&ifr, 0, sizeof( ifr ));
  strcpy(ifr.ifr_name, sInterface);
  Result = ioctl(SockRecvSession, SIOCGIFINDEX, (char *)&ifr);	
  if (Result < 0) {
	syslog(LOG_INFO, "FAILED(%d): ioctl(), %s(%d).", __LINE__,
													strerror(errno), errno);
	return(Result);
  } 
  memset(&XcvSockAddress, 0, sizeof(XcvSockAddress));
  XcvSockAddress.sll_family = PF_PACKET;
  XcvSockAddress.sll_protocol = htons(kPPPoE_Session_eType);
  XcvSockAddress.sll_ifindex = ifr.ifr_ifindex;
#else
  memset(&XcvSockAddress, 0, sizeof(XcvSockAddress));
  XcvSockAddress.sa_family = AF_INET;
  strcpy(XcvSockAddress.sa_data, sInterface);  	
#endif
  // This is tacky are the man pages wrong or the .h files.  I should not have 
  // to cast (struct sockaddr_ll *) to (struct sockaddr *) to use PF_PACKET
  // method of Linux 2.2 when doing a bind.
  Result = 
  bind(SockRecvSession, 
					(struct sockaddr *)&XcvSockAddress, sizeof(XcvSockAddress));
  if (Result < 0) {
	syslog(LOG_INFO, "FAILED(%d): bind(), %s(%d).", __LINE__, 
													strerror(errno), errno); 
    return(Result);
  }

  // Create socket to receive discovery packets from interesting interface.
#ifdef LINUX2_2
  XcvSockAddress.sll_protocol = htons(kPPPoE_Discovery_eType);
  SockRecvDiscovery = 
  socket(PF_PACKET, SOCK_RAW, htons(kPPPoE_Discovery_eType));
#else
  SockRecvDiscovery = 
  socket(AF_INET, SOCK_PACKET, htons(kPPPoE_Discovery_eType));
#endif
  if (SockRecvDiscovery < 0) {
	syslog(LOG_INFO, "FAILED(%d): socket(,, 0x%4.4X), %s(%d).", 
					__LINE__, kPPPoE_Discovery_eType, strerror(errno), errno);
    return(SockRecvDiscovery);
  }
  Result = bind(SockRecvDiscovery, 
					(struct sockaddr *)&XcvSockAddress, sizeof(XcvSockAddress));
  if (Result < 0) {
	syslog(LOG_INFO, "FAILED(%d): bind(), %s(%d).", __LINE__, 
													strerror(errno), errno);
    return(Result);
  }

  SockSend = SockRecvSession;

  // Get MAC address of ethernet adapter
  memset(&ifr, 0, sizeof( ifr ));
  strcpy(ifr.ifr_name, sInterface);
  Result = ioctl(SockSend, SIOCGIFHWADDR, (char *)&ifr);	
  if (Result < 0) {
	syslog(LOG_INFO, "FAILED(%d): ioctl(), %s(%d).", __LINE__,
													strerror(errno), errno);
	return(Result);
  } 
  if (ifr.ifr_hwaddr.sa_family == ARPHRD_ETHER) {
  	memcpy(&saddr, &ifr.ifr_hwaddr.sa_data, sizeof(saddr));
  }	else {
	syslog(LOG_INFO, "FAILED: %s is not an Ethernet interface.", sInterface);
    return( -1 );
  }

  return( 1 );
}


//
// Save Access Concentrator Name
//
void
PPPoE_SetACName( const char *name )
{
  gACNameLength = strlen( name );
  if( gACNameLength < sizeof(gACName) )
    {
	  memcpy( gACName, name, gACNameLength );
	  // DontCareFoundMask bits indicate what we do not care about
	  // It starts with all items set, we then clear then as we find we 
	  // need to look for them.
	  gACName[gACNameLength] = '\0';
	  DontCareFoundMask &= ~kFoundACName;
	}
  else
    {
	  fprintf( stderr, "\nAccess Concentrator Name too long.\n"
	    "Length must not exceed %d characters.\n", sizeof(gACName) );
	  gACNameLength=0;
	}
}

//
// Save Service Name
//
void
PPPoE_SetServiceName( const char *name )
{
  gServiceNameLength = strlen( name );
  if( gServiceNameLength < sizeof(gServiceName) )
    {
	  memcpy( gServiceName, name, gServiceNameLength );
	  gServiceName[gServiceNameLength] = '\0';
	  DontCareFoundMask &= ~kFoundServiceName;
	}
  else
    {
	  fprintf( stderr, "\nService Name too long.\n"
	    "Length must not exceed %d characters.\n", sizeof(gServiceName) );
	  gServiceNameLength=0;
	}
}

//
// Get a list of AC-Names
//
void
PPPoE_ViewACNames( void )
{
  struct timeval Timer;
  fd_set In_fd_set;

  PPPoE_SendPADI();					// This sets gState
  SET_STATE(kPPPoe_GatherACNames);	// Correct gState for our special operation

  WRITE_DLI( gXmitFrame, gXmitFrameLength );// Send Request

  FD_ZERO(&In_fd_set);
  FD_SET(0, &In_fd_set);	//stdin
  Timer.tv_sec = 4;
  Timer.tv_usec = 0;
  select(1, &In_fd_set, NULL, NULL, &Timer);//Wait a moment for all the replies
  SET_STATE(kPPPoE_LastState);
}

char sCopyright[]=
"Copyright (c) 1999,2000 by Network TeleSystems, Inc.\n"
"All rights reserved.\n"
"Internet http://www.nts.com\n\n"
"Please direct all support-related calls to your licensed Internet Service\n" 
"Provider.  NTS does not provide direct support.  NTS works through Internet\n"
"Service Providers that have licensed our software.\n";

struct sigaction	sa;

boolean bStrCmp(char *ps1, char *ps2)
{
	int Length = strlen(ps1);
	if(Length == strlen(ps2))
		return (memcmp(ps1, ps2, Length)==0);
    else
		return FALSE;
}

void LocalExit( int RetCode )
{
    if (PPPSendCount || PPPRecvCount)
		  syslog(LOG_INFO, 
		  		"Good packets sent, %d, and received, %d, by pppd.", 
		  		PPPSendCount, 
		  		PPPRecvCount);
    if (PppdFrameErrors)
    	  syslog(LOG_INFO, "Bad frames received from pppd %d.", PppdFrameErrors);
	syslog( LOG_INFO, "Exit." );
	closelog();
	exit(RetCode);
}

void LocalExitEarly(int RetCode, char *pString)
{
	if (pString) {
		fprintf(stderr, "\n%s\n", pString);
		syslog(LOG_INFO, pString);
	}
	syslog(LOG_INFO, "Exit.");
	closelog();
	exit(RetCode);
}

static void CatchSignal(int SignalNum, void(*handler)(int), int LineNo)
{
	int RetValue;
	sa.sa_handler = handler;
	RetValue = sigaction(SignalNum, &sa, NULL);
	if (RetValue < 0) {
	 	syslog(LOG_INFO, "FAILED(%d): sigaction(,,), %s(%d).", LineNo, 
												strerror(errno), errno); 
		LocalExit(RetValue);
	}
}

static void TerminateSignalHandler(int Signal)
{
	syslog(LOG_INFO, "Received signal, %d.", Signal);
	SET_STATE(kPPPoE_LastState);
}

static void HangUpSignalHandler(int Signal)
{
	int RetValue;
	
	if (gbShutDown)
		return;
	gbShutDown = TRUE;

	syslog(LOG_INFO, "Received %s.", Signal==1 ? "SIGHUP" : "SIGINT");

	//Allow 2 sec for pppd to send and receive a LCP:Terminate-ACK
	RetValue = alarm(2);
	if (RetValue < 0)
	 	syslog(LOG_INFO, "FAILED(%d): alarm(2), %s(%d).", __LINE__, 
														strerror(errno), errno); 
	if (RetValue > 0)
	 	syslog(LOG_INFO, "FAILED(%d): alarm(2) a previous Timer was running.",
														 __LINE__); 

}

static void AlarmSignalHandler(int Signal)
{
	syslog(LOG_INFO, "SUCCESS(%d): alarm signaled.", __LINE__); 
	SET_STATE(kPPPoE_LastState);
}

static void Usage(char *name)
{
	fprintf(stderr,
"\n%s Version: %u.%3.3u.%3.3u%s\n"
"\n%s\n"
"Usage:\n"
"  %s [ options ], where options are:\n"
"    <device>            Communicate to pppd through <device>.\n" 
"                        Defaults to stdio.\n"
"    -A <acname>         <acname> is the access concentrator to connect to.\n"
"    -I <ethernet>       <ethernet> is the name of the ethernet driver to\n"
"                        pass pppoe packets to.  Defaults to eth0.\n"
"    -S <service>        <service> the name of the service to connect to.\n"
"    -L                  List available AC names and service namess.\n"
"    -v                  Prints the version number of pppoe.\n"
"    sync | async        Use to match method pppd is using. Default is auto.\n"
"    boost               When non-zero, sets thread priority, range is 0 - 100.\n"
"                        Default is 75.\n"				
, name, MAJOR_VER, MINOR_VER, BUILD_NO, RELEASE_CANDIDATE, sCopyright, name);
}

//
//
//
int
main(int argc, char *argv[])
{
  char ErrorString[1024], PtyName[16];
//  struct link_int *link;
  pthread_t thread;
  pthread_attr_t thread_attr;
  struct sched_param our_sched_param;
  int argc_count, ErrorCode, Length;
  boolean bGetACNamesList=FALSE;

  openlog( "pppoed", (LOG_CONS | LOG_PID), LOG_DAEMON );
  syslog(LOG_INFO, "pppoed %u.%3.3u.%3.3u%s started", MAJOR_VER, MINOR_VER, 
		BUILD_NO, RELEASE_CANDIDATE);

  gProcessID = getpid();
  gProcessGroupID = getpgrp();

  sa.sa_flags = 0;
  sigemptyset(&sa.sa_mask);
  sigaddset(&sa.sa_mask, SIGHUP);
  sigaddset(&sa.sa_mask, SIGINT);
  sigaddset(&sa.sa_mask, SIGTERM);

  CatchSignal(SIGHUP, HangUpSignalHandler, __LINE__);		// Hangup
  CatchSignal(SIGINT, HangUpSignalHandler, __LINE__);		// Interrupt
  CatchSignal(SIGTERM, TerminateSignalHandler, __LINE__);	// Terminate
  CatchSignal(SIGALRM, AlarmSignalHandler, __LINE__);		// Terminate
  
  CatchSignal(SIGCHLD, SIG_IGN, __LINE__);		// Child 
  CatchSignal(SIGUSR1, SIG_IGN, __LINE__);		// pppd Toggle debug flag
  CatchSignal(SIGUSR2, SIG_IGN, __LINE__);		// pppd Reopen CCP
  // Apparently we can get a SIGPIPE when we call syslog, if syslogd has 
  // died and been restarted.  
  CatchSignal(SIGPIPE, SIG_IGN, __LINE__);

  // Process command line options 2ND
  //
  sInterface = "eth0";			// Assume defaults
  PtyName[0] = '\0';
  gSyncOption = FRAME_METHOD_AUTO;

  // 1ST argument must be pty
  for (argc_count = 1; argc_count < argc; argc_count++)
  {
      if (argv[argc_count][0] == '-' && strlen(argv[argc_count]) <= 2) {
		  switch (argv[argc_count][1]) {
		  case 'A':								// AC-Name
		      argc_count++;
		      if (argc_count < argc)
		          PPPoE_SetACName(argv[argc_count]);
		      break;
    	  case 'S':								// Service-Name
		      argc_count++;
		      if (argc_count < argc)
		          PPPoE_SetServiceName(argv[argc_count]);
		      break;
		  case 'I':								// interface device name
		      argc_count++;
		      if (argc_count < argc)
  		  	      Length = strlen( argv[argc_count] );
		  	  if (Length+1 <= 8)
		  	      sInterface = argv[argc_count];
		  	  else {
		  	      sprintf(ErrorString, 
				    "Network device name, \"%s\", is too long.", 
		  	  		argv[argc_count]);
		  	  	  LocalExitEarly(-1, ErrorString);
		  	  }
		      break;
		  case 'l':								// List AC Names responding
		  case 'L':								// List AC Names responding
		  	  bGetACNamesList=TRUE;
		  	  break;
		  case 'v':								// Version info
		  case 'V':								// Version info
		      fprintf(stderr, "\n%s Version: %u.%3.3u.%3.3u%s\n\n", argv[0],
		  	  				MAJOR_VER, MINOR_VER, BUILD_NO, RELEASE_CANDIDATE);
		  	  fprintf(stderr, sCopyright);
		      LocalExitEarly(0,NULL);	  
		  case 'h':
		  case 'H':
			  Usage(argv[0]);
		      LocalExitEarly(0,NULL);	  
		  case ' ':
		  case '\0':							// "-" assumes stdio
		      break;
		  default:
		  	  sprintf(ErrorString, "Unknown option, \"%s\".", 
					  					argv[argc_count]);	
		  	  LocalExitEarly(-1, ErrorString);
		      break;
		  }
	  } else if (argv[argc_count][0] == '/') {	// Must be a pty device name
		  Length = strlen(argv[argc_count]) ;
		  if (Length+1 > sizeof(PtyName)) {
		      sprintf(ErrorString, "Pty device name, \"%s\", is too long.", 
			  					argv[argc_count]);	
			  LocalExitEarly(-1, ErrorString);
		  }
		  memcpy(PtyName, argv[argc_count], Length+1);
		  if(PtyName[5] == 't') PtyName[5] = 'p';
		  if(PtyName[5] != 'p') {
			  sprintf(ErrorString, "Invalid pty device name, \"%s\".", 
			  					argv[argc_count]);
			  LocalExitEarly(-1, ErrorString);
		  }
	  }	else if (bStrCmp(argv[argc_count], "sync")) {
		  gSyncOption = FRAME_METHOD_SYNC;
          bSyncPPP = TRUE;
	  }	else if (bStrCmp(argv[argc_count], "async")) {
		  gSyncOption = FRAME_METHOD_ASYNC;
          bSyncPPP = FALSE;
	  }	else if (bStrCmp(argv[argc_count], "boost")) {
		  char *EndPtr;
		  argc_count++;
		  if (argc_count < argc) {
		     PriorityBoost = strtol(argv[argc_count], &EndPtr, 10);
			 if (*argv[argc_count] != '\0' && *EndPtr == '\0') {
			 	 if (PriorityBoost >= 0 && PriorityBoost <= 100)
					continue;
			 }
		  }
		  sprintf(ErrorString, 
				"The boost option, must be in the range of 0 to 100."); 
		  LocalExitEarly(-1, ErrorString);
	  }	else if (bStrCmp(argv[argc_count], "debug")) {
		  Debug=1;
	  } else {
		  sprintf(ErrorString, "Unknown option, \"%s\".", 
			  					argv[argc_count]);	
		  LocalExitEarly(-1, ErrorString);
	  }
  }

  if (Debug) {
	  syslog(LOG_INFO, "Built w/GLIBC Version: %d.%d", __GLIBC__,__GLIBC_MINOR__);
  }

  if (!bGetACNamesList) {
	  if (PtyName[0] == '\0' ) {
	  	  pppd_in_fd = 0;
		  pppd_out_fd = 1;
	  } else { 
      	  pppd_in_fd =  open(PtyName, O_RDWR|O_NOCTTY); //|O_SYNC
      	  if (pppd_in_fd < 0) {
      	  	  sprintf(ErrorString, "Could not open pty device, \"%s\". %s(%d).", 
  								PtyName, strerror(errno), errno);
      	  	  LocalExitEarly(-1, ErrorString);
		  }
      	  pppd_out_fd = dup(pppd_in_fd);
      	  if (pppd_out_fd < 0) {
      	  	  sprintf(ErrorString, "FAILED(%d): dup(), %s(%d).", 
   	  	  									__LINE__, strerror(errno), errno);
      	  	  LocalExitEarly(-1, ErrorString);
		  }
	  }
  }

  if (InitSockets() < 0) 
      LocalExit(1);

  if (PriorityBoost == 0) {
	  ErrorCode = pthread_create(&thread, NULL, DoReceivePacketThread, 0);
  } else {
	  int pmax, pmin;
  	  pmin = sched_get_priority_min(SCHED_FIFO);
	  pmax = sched_get_priority_max(SCHED_FIFO);
      our_sched_param.sched_priority=((pmax - pmin)*PriorityBoost)/100 + pmin;
      if (our_sched_param.sched_priority > pmax)
	  	  our_sched_param.sched_priority = pmax;
	  ErrorCode = sched_setscheduler(0, SCHED_FIFO, &our_sched_param);
      if (ErrorCode) { 
      	  syslog(LOG_INFO, "FAILED(%d): sched_setscheduler(,,), %s(%d).", 
   	  	  									__LINE__, strerror(errno), errno);
      	  LocalExit(1);
      }
	  if (Debug)
		  syslog(LOG_INFO, "sched_priority=%d  min=%d  max=%d.", 
							our_sched_param.sched_priority, pmin, pmax); 

      pthread_attr_init(&thread_attr);
	  our_sched_param.sched_priority += 1;
      if (our_sched_param.sched_priority > pmax)
	  	  our_sched_param.sched_priority = pmax;
	  
      ErrorCode = pthread_attr_setschedpolicy(&thread_attr, SCHED_FIFO);
      if (ErrorCode) { 
      	  syslog(LOG_INFO, "FAILED(%d): pthread_attr_setschedpolicy(,), %s(%d).", 
   	  	  									__LINE__, strerror(errno), errno);
      	  LocalExit(1);
      }
      ErrorCode = pthread_attr_setschedparam(&thread_attr, &our_sched_param);
      if (ErrorCode) { 
      	  syslog(LOG_INFO, "FAILED(%d): pthread_attr_setschedparam(,), %s(%d).", 
   	  	  									__LINE__, strerror(errno), errno);
      	  LocalExit(1);
      }
      ErrorCode = pthread_create(&thread, &thread_attr, DoReceivePacketThread, 0);
  }

  if (ErrorCode) { 
  	  syslog(LOG_INFO, "FAILED(%d): pthread_create(,,,), %s(%d).", 
	  	  									__LINE__, strerror(errno), errno);
	  LocalExit(1);
  }

  pthread_detach(thread);

  if (bGetACNamesList) {
  	  PPPoE_ViewACNames();
	  LocalExit(0);
  }

  if (!PPPoE_Discovery()) {
	  syslog(LOG_INFO,"Could not connect to requested network resource." ); 
      LocalExit(1);
  }
  
  Pppd2Ethernet();

  if (gSessionID) {  
	 // Send a PADT, when we have never sent or received an LCP Terminate ACK.
  	 if (!(gbXmtTerminateAck || gbRcvTerminateAck))
		PPPoE_SendPADT();
  }

  LocalExit(0);
}

