// CHGUSER.CPP
// ===========
//
//   Command line utility for setting up users for Serv-U.
//   Syntax:
//            chguser [-name=<username>] [-passwd=<password>] [-homedir=<path>]
//                    [-homeip=<homeip>] [-[access=<path,access>] ...]
//                    [-inifile=<path/name>]
//
//
//   Parameters containing spaces should be enclosed in quotation marks ("). Any
//   number of access rules is allowed, they are stored in the order found on the
//   command line. The 'access' part of an access rule can have the following
//   values (one or more):
//
//              R = read access (files)
//              W = write access (files)
//              M = modify/delete access (files)
//              C = create access (dirs)
//              D = delete access (dirs)
//              L = list access (dirs)
//              E = execute access (programs)
//              P = child dirs inherit rule
//
//
//          Author: Rob Beckers
//            Date: 11-MAY-97
//   Last revision: 11-MAY-97
//     Revision nr: 1
//
//   Revision history:
//

#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "passwd.h"
#include "chguser.h"

void main(int argc,char* argv[])
// ******
// Main routine of progam.
// ******
{
  string Name,PassWord,HomeDir,HomeIP,IniFile;  // parameters
  LSTSTRINGS AccessLst;                         // list of access rules

  // defaults
  IniFile=argv[0];                              // use current path as base for default .ini
  IniFile.remove(IniFile.find_last_of("\\")+1); // strip .exe name
  IniFile.append(FTP_DEFINIFILE);               // default name

  // only if there are parameters
  if (argc<2) {
    printf("No parameter(s) specified\n\n");
    printf("chguser [-name=<username>] [-passwd=<password>] [-homedir=<path>]\n");
    printf("        [-homeip=<homeip>] [-[access=<path,access>] ...]\n");
    printf("        [-inifile=<path/name>]\n\n");
    printf("Parameters containing spaces should be enclosed in quotation marks (\"). Any\n");
    printf("number of access rules is allowed, they are stored in the order found on the\n");
    printf("command line. The 'access' part of an access rule can have the following\n");
    printf("values (one or more):\n\n");
    printf("     R = read access (files)\n");
    printf("     W = write access (files)\n");
    printf("     M = modify/delete access (files)\n");
    printf("     C = create access (dirs)\n");
    printf("     D = delete access (dirs)\n");
    printf("     L = list access (dirs)\n");
    printf("     E = execute access (programs)\n");
    printf("     P = child dirs inherit rule\n");
    return;
  }

  // parse parameters
  string::set_case_sensitive(0);        // comparisons not case sensitive for options
  for (int n=1;n<argc;n++) {
    string Arg=argv[n];
    int Idx;
    if ((Idx=Arg.find("="))!=NPOS) {
      string Option=Arg.substr(0,Idx);  // split up option and value
      string Value=Arg.substr(Idx+1);

      // user name
      if (Option==FTP_NAMEOPT) {
        Name=Value;
      }

      // password
      else if (Option==FTP_PASSWDOPT) {
        PassWord=Value;
      }

      // home directory
      else if (Option==FTP_HOMEDIROPT) {
        HomeDir=Value;
      }

      // home IP address
      else if (Option==FTP_HOMEIPOPT) {
        HomeIP=Value;
      }

      // access rule
      else if (Option==FTP_ACCESSOPT) {
        AccessLst.Add(Value);
      }

      // .ini file
      else if (Option==FTP_INIFILEOPT) {
        IniFile=Value;
      }
    }
  }

  // quit if there's no user name
  if (!Name.length()) {
    printf("No user name specified\n");
    return;
  }

  // if we have an IP number look up the host number in the IP list
  if (HomeIP.length()) {
    RHomeIPStr HomeStruc;
    LSTHOMESTR HomeList;		// list of home-IPs
    LoadIPHomes(HomeList,IniFile);      // load a list with all IP homes
    HomeStruc.HomeIP=HomeIP;            // IP we want to look for
    if (HomeList.Find(HomeStruc)) {     // look for our IP number
      char NumStr[20];                  // modify user name to reflect home IP
      Name=Name+"@"+itoa(HomeStruc.No,NumStr,10);
    }
    else {
      printf("Home IP not found\n");
      return;
    }
  }

  // create the user name for .ini file access
  char User[100];
  strcpy(User,FTP_USER);
  strcat(User,Name.c_str());

  // if we only have a user name and no parameters for that user delete the user
  if (!HomeDir.length() && !PassWord.length() && !AccessLst.NrItems()) {
    WritePrivateProfileString(User,NULL,NULL,IniFile.c_str());
    WritePrivateProfileString(NULL,NULL,NULL,IniFile.c_str()); // flush file
    return;
  }

  // if we have one or more access rules delete the old rules
  if (AccessLst.NrItems()) {
    int Count=1; int n;
    char Rule[255]; char Entry[50];
    do {
      sprintf(Entry,"%s%d",FTP_ACCESS,Count);
      if ((n=GetPrivateProfileString(User,Entry,"",Rule,254,IniFile.c_str()))!=0) {
        WritePrivateProfileString(User,Entry,NULL,IniFile.c_str());
        Count++;
      }
    }
    while (n);
  }

  // write the new user info
  if (PassWord.length()) {              // encrypt & write password
    WritePrivateProfileString(User,FTP_PASS,MakePassword(PassWord.c_str()),IniFile.c_str());
  }
  if (HomeDir.length()) {               // write home directory
    WritePrivateProfileString(User,FTP_HOME,HomeDir.c_str(),IniFile.c_str());
  }
  if (AccessLst.NrItems()) {            // write access rules
    string Rule; int Count=1; char Entry[50];
    AccessLst.GetFirst(Rule);
    do {
      sprintf(Entry,"%s%d",FTP_ACCESS,Count);
      WritePrivateProfileString(User,Entry,Rule.c_str(),IniFile.c_str());
      Count++;
    }
    while (AccessLst.GetNext(Rule));
  }

  // flush .ini file changes to disk
  WritePrivateProfileString(NULL,NULL,NULL,IniFile.c_str());
}

int operator==(const RHomeIPStr& Str1,const RHomeIPStr& Str2)
// ******
// Comparison operator for two RHomeIPStr's.
// Used by RSimpleList.
// ******
{
  // compare the IP numbers
  if (Str1.HomeIP==Str2.HomeIP) return(1);
  else return(0);
}

void LoadIPHomes(LSTHOMESTR& HomeList,const string& IniFile)
// ******
// Load list of IP homes for multi-homed support.
// ******
{
  HomeList.Flush();			// clean out the home list

  // read the whole home section
#ifdef __WIN32__
  #define _IPBUFSIZE 10000  		// good for about 1800 IP numbers
#else
  #define _IPBUFSIZE 1000               // allows 180 IP numbers
#endif
  char Buff[_IPBUFSIZE]; int n;
  n=GetPrivateProfileString(FTP_IPHOMES,NULL,"",Buff,_IPBUFSIZE,IniFile.c_str());
  if (!n) return;

  // parse the section
  char IPString[25];
  char* pCurr=Buff;			// pointer to current entry
  while (TRUE) {
    if (strnicmp(pCurr,FTP_IP,strlen(FTP_IP))==0) {
      int No=0;
      if (sscanf(pCurr+strlen(FTP_IP),"%d",&No)==1) {

        // get entry
        if (GetPrivateProfileString(FTP_IPHOMES,pCurr,"",IPString,25,IniFile.c_str())) {
          RHomeIPStr HomeStruc;
          HomeStruc.No=No;
          HomeStruc.HomeIP=IPString;
          HomeList.Add(HomeStruc);	// add to list
        }
      }
    }

    // go to next section
    if ((pCurr=strchr(pCurr,'\0'))==NULL) break;
    pCurr++;                    	// check for end of section
    if (pCurr[0]=='\0') break;
  }
}

