program SplitJoin;


//Project:         SplitJoin.exe
//Description:     Utility to split and rejoin large files
//Version:         1.3.3
//Last Updated:    24-JAN-2002
//Compiler:        Delphi 2 - 6
//Author:          Angus Johnson, ajohnson@rpi.net.au
//Copyright:        2001-2002 Angus Johnson
//Feedback:        Please email comments, suggestions to the author.

//Updated 18-MAR-2001:
//Bug with Multiselect in OpenFileDialog() fixed.
//Several minor interface improvements.

//Updated 27-MAR-2001:
//1. Help Contents dialog added
//2. strings moved to resource section (aids translation etc)
//3. joined files now checked for errors with 32bit checksum
//4. Statusbar added to feedback file splitting/joining progress, errors etc.
//5. File datetime preserved.
//6. Added a web address for SplitJoin on the AboutDialog (plus hand cursor etc.)
//7. Bugfix: forgot to store path of joining files.

//Updated 4-APR-2001:
//Escape key interrupts splitting/joining operation
//Compiled using Delphi 2 and Packed with UPX -> 30kb executable!
//(UPX -> http://wildsau.idv.uni-linz.ac.at/mfx/upx.html)

//Updated 9-SEP-2001:
//Delphi 6 compatible
//Open Folder now has Initial directory set

//Updated 24-JAN-2002:
//Minor bug fixes for compiling with Delphi 6

uses
  Windows,
  messages,
  CommDlg,
  SysUtils,
  CommCtrl,
  ShlObj,
  ShellApi,
  Richedit;                               

{$R splitjoin2.res}


const
  strFloppySize = '1440';
  minAllowedSplitSize = 120; //min 120kb
  strBlank = '';

  idStatusbar = 8;
  idtab = 10;
  menuHelp = 1001; //also accelerator
  menuAbout = 1002;
  accelSkip = 2001;
  accelJoin = 2002;

  //split/join dialogs...
  idOKBtn = 9;
  idFileEdit = 11;
  idFileBrowse = 12;
  idSaveToEdit = 13;
  idSaveToBrowse = 14;
  idFloppy = 15;
  idSizeEdit = 16;
  idMakeBatch = 17;
  idFileCountStatic = 18;
  idListbox = 20;

  //about dialog...
  idBoldStatic = 101;
  idWebStatic = 102;
  //help dialog...
  idRichedit = 31;

  //stringtable constants...
  sAllFiles = 1;          //'All Files'
  sSelectSaveToFolder = 2;//'Select the folder where the file parts are to be saved ...'
  sOnlyNumberedFiles = 3; //'Only numbered files can be joined!'
  sSelectSplitFile = 4;   //'Select the file to split...'
  sSelectJoinFiles = 5;   //'Select any one of the files to join ...'
  sNewJoinFilename = 6;   //'Enter the name for the joined file ...'
  sSplit = 7;             //'&Split';
  sJoin = 8;              //'&Join';
  sFontface = 9;          //'Arial'
  sChecksumOK = 11;
  sChecksumFail = 12;
  sSplittingFiles = 16;
  sFileDoesntExist = 17;
  sFolderDoesntExist = 18;
  sCantOpenFile = 19;
  sCantCreateFile = 20;
  sJoiningFiles = 21;
  sSplitFinished = 22;
  sJoinFinished = 23; //(Use this when no Checksum file found to test.)
  sCantReadWriteFile = 24;
  sCreatingChecksum = 25;
  sCheckingChecksum = 26;
  sOutOfMemory = 27;
  sCancelled = 28;

type
  TTabShowing = (tsSplit, tsJoin);

  {this structure definition changes between delphi versions hence
  to enable consistency between compilers...}
  TOpenFilename = packed record
    lStructSize: DWORD;
    hWndOwner: HWND;
    hInstance: HINST;
    lpstrFilter: PAnsiChar;
    lpstrCustomFilter: PAnsiChar;
    nMaxCustFilter: DWORD;
    nFilterIndex: DWORD;
    lpstrFile: PAnsiChar;
    nMaxFile: DWORD;
    lpstrFileTitle: PAnsiChar;
    nMaxFileTitle: DWORD;
    lpstrInitialDir: PAnsiChar;
    lpstrTitle: PAnsiChar;
    Flags: DWORD;
    nFileOffset: Word;
    nFileExtension: Word;
    lpstrDefExt: PAnsiChar;
    lCustData: LPARAM;
    lpfnHook: function(Wnd: HWND; Msg: UINT; wParam: WPARAM; lParam: LPARAM): UINT stdcall;
    lpTemplateName: PAnsiChar;
    pvReserved: Pointer;
    dwReserved: DWORD;
    FlagsEx: DWORD;
  end;

  function GetOpenFileName(var OpenFile: TOpenFilename): Bool; stdcall;
    external 'comdlg32.dll'  name 'GetOpenFileNameA';
  function GetSaveFileName(var OpenFile: TOpenFilename): Bool; stdcall;
    external 'comdlg32.dll'  name 'GetSaveFileNameA';

var
  dlg: THandle;
  tab: THandle;
  accel: THandle;
  SplitDlg, JoinDlg: THandle;
  TabShowing: TTabShowing;
  boldFont, webFont: HFont;
  joinSrcPath: string;
  osVersionInfo: TOSVersionInfo;

//---------------------------------------------------------------------
//---------------------------------------------------------------------

//GetLongFilename() is really only needed when files are dropped
//onto the SplitJoin.exe icon in Explorer, as Explorer still passes the
//8.3 filename format in the commandline...
function GetLongFilename(const fn: string): string;
var
  desktop: IShellFolder;
  OlePath: array[0..MAX_PATH] of WideChar;
  pidl: PItemIDList;
  dummy1,dummy2: ULONG;
  StrRet: TStrRet;
begin
  result := fn;
  if (length(fn) < 4) or (fn[2] <> ':') then exit;
  if SHGetDesktopFolder(desktop) <> NOERROR then exit;
  StringToWideChar( fn, OlePath, MAX_PATH );
  if desktop.ParseDisplayName(0,nil,OlePath,dummy1,pidl,dummy2) <> NOERROR then
    exit;
  if desktop.GetDisplayNameOf(pidl,SHGDN_FORPARSING,StrRet) <> NOERROR then
    exit;
  case StrRet.uType of
    STRRET_WSTR: Result:=WideCharToString(StrRet.pOleStr);
    STRRET_OFFSET: Result:=PChar(UINT(Pidl)+StrRet.uOffset);
    STRRET_CSTR: Result:=StrRet.cStr;
  End;
end;
//--------------------------------------------------------------------------

function OpenFileDialog(Owner: THandle; const Caption, Filter,
  InitialDir: string; var Filename: string; MultiSelect: boolean): boolean;
var
  ofn: TOpenFilename;
  filepath: array [0..1024] of char;
  i: integer;
  newfilter: string;
begin
  strPCopy(filepath,Filename); //nb: ofn.lpstrFile must be initialized (or #0)

  //nb: assumes only a single filter here...
  i := pos('|',Filter);
  if i > 0 then
  begin
    newfilter  := copy(Filter,1,i-1)+#0+copy(filter,i+1,255)+#0#0;
  end;
  if newfilter = '' then
    newfilter  := loadstr(sAllFiles)+#0'*.*'#0#0;


  fillchar(ofn,sizeof(ofn),0);
  //before win2000 & WinME the last 3 fields of TOpenFilename are not supported
  if ((osVersionInfo.dwMajorVersion >= 5) and
      (osVersionInfo.dwPlatformId = VER_PLATFORM_WIN32_NT)) {win2000 or winXP} or
      ((osVersionInfo.dwMajorVersion >= 4) and
      (osVersionInfo.dwPlatformId = VER_PLATFORM_WIN32_WINDOWS) and
      (osVersionInfo.dwMinorVersion >= 90)) {winME} then
    ofn.lStructSize := sizeof(TOpenFilename) else
    ofn.lStructSize := sizeof(TOpenFilename) -12;

  ofn.hWndOwner := Owner;
  ofn.lpstrFilter := pchar(newfilter);
  ofn.nFilterIndex := 1;
  ofn.lpstrFile := pchar(@filepath[0]);
  ofn.nMaxFile := MAX_PATH;
  ofn.lpstrFileTitle := nil;

  ofn.lpstrInitialDir := pchar(InitialDir);
  ofn.lpstrTitle := pchar(caption);
  ofn.Flags := OFN_FILEMUSTEXIST or OFN_PATHMUSTEXIST or OFN_HIDEREADONLY;
  if MultiSelect then
    ofn.Flags := ofn.Flags or OFN_ALLOWMULTISELECT or OFN_EXPLORER;
  result := GetOpenFileName(ofn);
  if not result then exit;
  //if MultiSelect then filepath can have 2 general forms:
  //  1. 'path&name'#0#0 or
  //  2. 'path'#0'name1'#0'name2'#0' ..nameN'#0#0
  if MultiSelect then
  begin
    //here, we're just interested in the first filename, so ...
    Filename := pchar(@filepath[strlen(filepath)+1]);
    if Filename = '' then
      Filename := filepath else
      Filename := format('%s\%s',[filepath, Filename]);
  end
  else
    Filename := filepath;
end;
//--------------------------------------------------------------------------

function SaveFileDialog(Owner: THandle; const Caption, Filter,
  InitialDir: string; var Filename: string): boolean;
var
  ofn: TOPENFILENAME;
  filpath: array [0..MAX_PATH] of char;
  filname: array [0..MAX_PATH] of char;
  i: integer;
  newfilter: string;
begin
  fillchar(ofn,sizeof(ofn),0);
  //before win2000 & WinME the last 3 fields of TOpenFilename are not supported
  if ((osVersionInfo.dwMajorVersion >= 5) and
      (osVersionInfo.dwPlatformId = VER_PLATFORM_WIN32_NT)) {win2000 or winXP} or
      ((osVersionInfo.dwMajorVersion >= 4) and
      (osVersionInfo.dwPlatformId = VER_PLATFORM_WIN32_WINDOWS) and
      (osVersionInfo.dwMinorVersion >= 90)) {winME} then
    ofn.lStructSize := sizeof(TOpenFilename) else
    ofn.lStructSize := sizeof(TOpenFilename) -12;

  strPCopy(filpath,Filename); //nb: ofn.lpstrFile must be initialized (or #0)

  //nb: assumes only a single filter here...
  i := pos('|',Filter);
  if i > 0 then
  begin
    newfilter  := copy(Filter,1,i-1)+#0+copy(filter,i+1,255)+#0#0;
  end;
  if newfilter = '' then
    newfilter  := loadstr(sAllFiles)+#0'*.*'#0#0;

  ofn.hWndOwner := Owner;
  ofn.lpstrFilter := pchar(newfilter);
  ofn.nFilterIndex := 1;
  ofn.lpstrFile := pchar(@filpath);
  ofn.nMaxFile := MAX_PATH;
  ofn.lpstrFileTitle := pchar(@filname);
  ofn.nMaxFileTitle := MAX_PATH;

  ofn.lpstrInitialDir := pchar(InitialDir);
  ofn.lpstrTitle := pchar(caption);
  ofn.Flags := OFN_OVERWRITEPROMPT or OFN_PATHMUSTEXIST or OFN_HIDEREADONLY;
  result := GetSaveFileName(ofn);
  if result then Filename := filpath;
end;
//--------------------------------------------------------------------------

function StripPathSlash(const path: string): string;
var
  i: integer;
begin
  result := path;
  i := length(path);
  if (i > 3) and (path[i] = '\') then setlength(result,i-1);
end;
//---------------------------------------------------------------------

//BrowseProc() is used by GetFolder() to set the initial folder and display
//the path for the currently selected folder.
function BrowseProc(hwnd: HWnd; uMsg: integer; lParam, lpData: LPARAM): integer; stdcall;
var
  Dir: array[0..MAX_PATH] of char;
begin
  case uMsg of
    BFFM_INITIALIZED:
      begin
        SendMessage(hwnd, BFFM_SETSTATUSTEXT,0, lpData);
        SendMessage(hwnd, BFFM_SETSELECTION, 1, lpData);
      end;
    BFFM_SELCHANGED:
      if(SHGetPathFromIDList(PItemIDList(lParam), Dir)) then
        SendMessage(hwnd, BFFM_SETSTATUSTEXT,0, integer(@Dir[0]));
  end;
  result := 0;
end;
//---------------------------------------------------------------------

procedure CoTaskMemFree(pv: Pointer); stdcall; external 'ole32.dll' name 'CoTaskMemFree';

function GetFolder(OwnerHdl: THandle; var Folder: string): boolean;
var
  displayname: array[0..MAX_PATH] of char;
  bi: TBrowseInfo;
  pidl: PItemIdList;
begin
  bi.hWndOwner := OwnerHdl;
  bi.pIDLRoot := nil;
  bi.pszDisplayName := pchar(@displayname[0]);
  bi.lpszTitle := pchar(loadstr(sSelectSaveToFolder));
  bi.ulFlags := BIF_RETURNONLYFSDIRS or BIF_STATUSTEXT;
  if Folder = '' then
  begin
    bi.lpfn := nil;
    bi.lParam := 0;
  end else
  begin
    Folder := StripPathSlash(Folder);
    bi.lpfn := @BrowseProc;
    bi.lParam := integer(pchar(Folder));
  end;
  bi.iImage := 0;
  pidl := SHBrowseForFolder(bi);
  result := pidl <> nil;
  if not result then exit;
  try
    result := SHGetPathFromIDList(pidl,pchar(@displayname[0]));
    Folder := displayname;
  finally
    CoTaskMemFree(pidl);
  end;
end;
//--------------------------------------------------------------------------

function IsOKToSplit: boolean;
var
  buff: array[0..1] of char;
begin
  sendDlgItemMessage(SplitDlg, idFileEdit, WM_GETTEXT,2,longint(@buff[0]));
  result := buff[0] <> #0;
  if not result then exit;
  sendDlgItemMessage(SplitDlg, idSaveToEdit, WM_GETTEXT,2,longint(@buff[0]));
  result := buff[0] <> #0;
end;
//--------------------------------------------------------------------------

function IsOKToJoin: boolean;
var
  buff: array[0..1] of char;
begin
  result := sendDlgItemMessage(JoinDlg, idListbox, LB_GETCOUNT,0,0) > 1;
  if not result then exit;
  sendDlgItemMessage(JoinDlg, idSaveToEdit, WM_GETTEXT,2,longint(@buff[0]));
  result := buff[0] <> #0;
end;
//--------------------------------------------------------------------------

procedure SetStatusbarText(const str: string);
begin
  sendDlgItemMessage( dlg, idStatusbar, WM_SETTEXT, 0, longint(pchar(' '+str)));
end;
//--------------------------------------------------------------------------

procedure ShowTheTab(TabToShow: TTabShowing);
begin
  if TabToShow = TabShowing then exit;
  TabShowing := TabToShow;
  if TabShowing = tsSplit then
  begin
    ShowWindow(SplitDlg,SW_SHOW);
    ShowWindow(JoinDlg,SW_HIDE);
    SetFocus(SplitDlg);
    Postmessage(tab,TCM_SETCURSEL,0,0);
  end else
  begin
    ShowWindow(SplitDlg,SW_HIDE);
    ShowWindow(JoinDlg,SW_SHOW);
    SetFocus(JoinDlg);
    Postmessage(tab,TCM_SETCURSEL,1,0);
  end;
  SetStatusbarText('');
end;
//--------------------------------------------------------------------------

function FillJoinListbox(const Filename: string): boolean;
var
  i: integer;
  filterStr: string;
  ffHdl: THandle;
  wfd: TWIN32FINDDATA;
begin
  result := false;
  sendDlgItemMessage(JoinDlg, idListbox, WM_SETREDRAW, 0, 0);
  try

    //empty the listbox...
    i := sendDlgItemMessage(JoinDlg, idListbox, LB_GETCOUNT,0,0);
    while i > 0 do
      i := sendDlgItemMessage(JoinDlg, idListbox, LB_DELETESTRING,0,0);

    //nb: 'joinSrcPath' is global as is needed in JoinFiles() too ...
    joinSrcPath := extractfilepath(Filename);
    if (joinSrcPath <> '') and
      (joinSrcPath[length(joinSrcPath)] <> '\') then
        joinSrcPath := joinSrcPath + '\';

    //get the findfile filter...
    //nb: Numbering does NOT have to be at the end of the filename
    //    in order to rejoin file parts even though that is how
    //    "SplitJoin" does name the split file parts.
    filterStr := extractfilename(Filename);
    i := length(filterStr);
    while (i > 0) and not (filterStr[i] in ['0'..'9']) do dec(i);
    if i = 0 then
    begin
      sendDlgItemMessage(JoinDlg, idListbox,
        LB_ADDSTRING, 0, longint(pchar(loadstr(sOnlyNumberedFiles))));
      exit;
    end;

    while (i > 0) and (filterStr[i] in ['0'..'9']) do
    begin
      filterStr[i] := '?';
      dec(i);
    end;
    filterStr := joinSrcPath + filterStr;

    //fill the listbox...
    ffHdl := findfirstfile(pchar(filterStr),wfd);
    if ffHdl = INVALID_HANDLE_VALUE then exit;
    try
      i := strlen(wfd.cFileName); //all related files must be the same length
      repeat
        //if its not the checksum file add it to the list...
        if pchar(@wfd.cFileName[i-4]) <> '.999' then
          sendDlgItemMessage(JoinDlg, idListbox,
            LB_ADDSTRING, 0, longint(pchar(ExtractFilename(wfd.cFileName))));
      until not FindNextFile(ffHdl,wfd);
    finally
      windows.FindClose(ffHdl);
    end;

  finally
    sendDlgItemMessage(JoinDlg, idListbox, WM_SETREDRAW, 1, 0);
  end;
  //count the number of file parts...
  i := sendDlgItemMessage(JoinDlg, idListbox, LB_GETCOUNT,0,0);
  if i = 1 then
  begin
    sendDlgItemMessage(JoinDlg, idListbox, LB_DELETESTRING, 0, 0);
    sendDlgItemMessage(JoinDlg, idListbox,
      LB_ADDSTRING, 0, longint(pchar(loadstr(sOnlyNumberedFiles))));
  end
  else
  begin
    SetFocus(GetDlgItem(JoinDlg,idSaveToBrowse));
    //this is a workaround otherwise Join's SaveAs Browse button wont
    // respond to the <enter> key if it receives initial focus!!! ...
    postmessage(JoinDlg,WM_NEXTDLGCTL,0,0);
    postmessage(JoinDlg,WM_NEXTDLGCTL,1,0);
    result := true;
  end;
end;
//--------------------------------------------------------------------------

//nb: GetDlgItemCaption() assumes the maximum length of the caption = MAX_PATH
function GetDlgItemCaption(hDlg: THandle; ItemId: integer): string;
begin
  setLength(result,MAX_PATH);
  setLength(result, sendDlgItemMessage(hDlg, ItemId,
    WM_GETTEXT,MAX_PATH,longint(pchar(result))));
end;
//--------------------------------------------------------------------------

function GetSplitSize: integer;
var
  str: string;
begin
  //get the max size of the split files ...
  str := GetDlgItemCaption(SplitDlg,idSizeEdit);
  result := strtointdef(str,0);
  if result < minAllowedSplitSize then result := 0;
end;
//--------------------------------------------------------------------------

function DirectoryExists(const Name: string): Boolean;
var
  Code: Integer;
begin
  Code := GetFileAttributes(PChar(Name));
  Result := (Code <> -1) and (FILE_ATTRIBUTE_DIRECTORY and Code <> 0);
end;
//--------------------------------------------------------------------------

//To enable the <Escape> key to stop splitting or joining...
function EscapePressed: boolean;
var
  Msg: TMsg;
begin
  while PeekMessage(Msg, 0, WM_KEYDOWN, WM_KEYDOWN, PM_REMOVE) do
    if (Msg.wParam = VK_ESCAPE) then
    begin
      SetStatusbarText(loadStr(sCancelled));
      result := true;
      exit;
    end;
  result := false;
end;
//--------------------------------------------------------------------------

const
  table:  ARRAY[0..255] OF DWORD =
 ($00000000, $77073096, $EE0E612C, $990951BA,
  $076DC419, $706AF48F, $E963A535, $9E6495A3,
  $0EDB8832, $79DCB8A4, $E0D5E91E, $97D2D988,
  $09B64C2B, $7EB17CBD, $E7B82D07, $90BF1D91,
  $1DB71064, $6AB020F2, $F3B97148, $84BE41DE,
  $1ADAD47D, $6DDDE4EB, $F4D4B551, $83D385C7,
  $136C9856, $646BA8C0, $FD62F97A, $8A65C9EC,
  $14015C4F, $63066CD9, $FA0F3D63, $8D080DF5,
  $3B6E20C8, $4C69105E, $D56041E4, $A2677172,
  $3C03E4D1, $4B04D447, $D20D85FD, $A50AB56B,
  $35B5A8FA, $42B2986C, $DBBBC9D6, $ACBCF940,
  $32D86CE3, $45DF5C75, $DCD60DCF, $ABD13D59,
  $26D930AC, $51DE003A, $C8D75180, $BFD06116,
  $21B4F4B5, $56B3C423, $CFBA9599, $B8BDA50F,
  $2802B89E, $5F058808, $C60CD9B2, $B10BE924,
  $2F6F7C87, $58684C11, $C1611DAB, $B6662D3D,

  $76DC4190, $01DB7106, $98D220BC, $EFD5102A,
  $71B18589, $06B6B51F, $9FBFE4A5, $E8B8D433,
  $7807C9A2, $0F00F934, $9609A88E, $E10E9818,
  $7F6A0DBB, $086D3D2D, $91646C97, $E6635C01,
  $6B6B51F4, $1C6C6162, $856530D8, $F262004E,
  $6C0695ED, $1B01A57B, $8208F4C1, $F50FC457,
  $65B0D9C6, $12B7E950, $8BBEB8EA, $FCB9887C,
  $62DD1DDF, $15DA2D49, $8CD37CF3, $FBD44C65,
  $4DB26158, $3AB551CE, $A3BC0074, $D4BB30E2,
  $4ADFA541, $3DD895D7, $A4D1C46D, $D3D6F4FB,
  $4369E96A, $346ED9FC, $AD678846, $DA60B8D0,
  $44042D73, $33031DE5, $AA0A4C5F, $DD0D7CC9,
  $5005713C, $270241AA, $BE0B1010, $C90C2086,
  $5768B525, $206F85B3, $B966D409, $CE61E49F,
  $5EDEF90E, $29D9C998, $B0D09822, $C7D7A8B4,
  $59B33D17, $2EB40D81, $B7BD5C3B, $C0BA6CAD,

  $EDB88320, $9ABFB3B6, $03B6E20C, $74B1D29A,
  $EAD54739, $9DD277AF, $04DB2615, $73DC1683,
  $E3630B12, $94643B84, $0D6D6A3E, $7A6A5AA8,
  $E40ECF0B, $9309FF9D, $0A00AE27, $7D079EB1,
  $F00F9344, $8708A3D2, $1E01F268, $6906C2FE,
  $F762575D, $806567CB, $196C3671, $6E6B06E7,
  $FED41B76, $89D32BE0, $10DA7A5A, $67DD4ACC,
  $F9B9DF6F, $8EBEEFF9, $17B7BE43, $60B08ED5,
  $D6D6A3E8, $A1D1937E, $38D8C2C4, $4FDFF252,
  $D1BB67F1, $A6BC5767, $3FB506DD, $48B2364B,
  $D80D2BDA, $AF0A1B4C, $36034AF6, $41047A60,
  $DF60EFC3, $A867DF55, $316E8EEF, $4669BE79,
  $CB61B38C, $BC66831A, $256FD2A0, $5268E236,
  $CC0C7795, $BB0B4703, $220216B9, $5505262F,
  $C5BA3BBE, $B2BD0B28, $2BB45A92, $5CB36A04,
  $C2D7FFA7, $B5D0CF31, $2CD99E8B, $5BDEAE1D,

  $9B64C2B0, $EC63F226, $756AA39C, $026D930A,
  $9C0906A9, $EB0E363F, $72076785, $05005713,
  $95BF4A82, $E2B87A14, $7BB12BAE, $0CB61B38,
  $92D28E9B, $E5D5BE0D, $7CDCEFB7, $0BDBDF21,
  $86D3D2D4, $F1D4E242, $68DDB3F8, $1FDA836E,
  $81BE16CD, $F6B9265B, $6FB077E1, $18B74777,
  $88085AE6, $FF0F6A70, $66063BCA, $11010B5C,
  $8F659EFF, $F862AE69, $616BFFD3, $166CCF45,
  $A00AE278, $D70DD2EE, $4E048354, $3903B3C2,
  $A7672661, $D06016F7, $4969474D, $3E6E77DB,
  $AED16A4A, $D9D65ADC, $40DF0B66, $37D83BF0,
  $A9BCAE53, $DEBB9EC5, $47B2CF7F, $30B5FFE9,
  $BDBDF21C, $CABAC28A, $53B39330, $24B4A3A6,
  $BAD03605, $CDD70693, $54DE5729, $23D967BF,
  $B3667A2E, $C4614AB8, $5D681B02, $2A6F2B94,
  $B40BBE37, $C30C8EA1, $5A05DF1B, $2D02EF8D);  

//PKZip compliant 32bit CRC algorithm
//Algorithm courtesy of Earl F. Glynn, and used with his kind permission ...
//(http://www.efg2.com/Lab/Mathematics/CRC.htm)
function CalcCRC32(p: pointer; ByteCount: dword): dword;
var
  i: dword;
  q: Pbyte;
begin
  q := p;
  result := $FFFFFFFF;
  for i := 0 to ByteCount-1 do
  begin
    result := (result shr 8) xor table[ q^ xor (result and $000000ff) ];
    inc(q);

    //check if escape pressed after each Mb parsed ...
    //this slows the algorithm a little but is necessary to allow the user to
    //break out of this function as it may take some time with very large files
    if (i mod $100000 = 0) and EscapePressed then
    begin
      result := $FFFFFFFF;
      exit;
    end;

  end;
  result := not result;
end;
//--------------------------------------------------------------------------

function GetChecksum(const FileHandle: THandle): DWORD;
var
  mapHdl: THandle;
  memPtr: Pointer;
begin
  result := $FFFFFFFF; //ie: assume error
  memPtr := nil;
  mapHdl := 0;
  try
    mapHdl := CreateFileMapping(FileHandle, nil, PAGE_READONLY, 0, 0, nil);
    if mapHdl = 0 then exit;
    memPtr := MapViewOfFile(mapHdl, FILE_MAP_READ, 0, 0, 0);
    if memPtr = nil then exit;
    result := CalcCRC32(memPtr, GetFileSize(FileHandle, nil));
  finally
    if Assigned(memPtr) then UnmapViewOfFile(memPtr);
    if mapHdl <> 0 then CloseHandle (mapHdl);
  end;
end;
//---------------------------------------------------------------------

procedure SplitFiles;
var
  cnt, SplitSize, ReadSize: integer;
  Checksum: dword;
  readFn, writeFn, splitFn: string;
  fReadHdl, fWriteHdl: THandle;
  fModified: TFileTime;
  buffer: pchar;
  oldCursor: HCursor;
begin
  //disable 'OK' button...
  EnableWindow(GetDlgItem(splitDlg,idOKBtn), false);

  SplitSize := GetSplitSize;
  if SplitSize < minAllowedSplitSize then exit; //something wrong!!
  SplitSize := SplitSize*1024; //nb: was in Kb's

  //get the name of the file to split...
  readFn := GetDlgItemCaption(SplitDlg, idFileEdit);
  if not fileexists(readFn) then
  begin
    SetStatusbarText(loadStr(sFileDoesntExist));
    beep;
    exit; //something wrong!!
  end;

  //get the 'target' folder (where split files are to be saved) ...
  writeFn  := GetDlgItemCaption(SplitDlg, idSaveToEdit);
  if not directoryexists(writeFn) then
  begin
    SetStatusbarText(loadStr(sFolderDoesntExist));
    beep;
    exit; //something wrong!!
  end;
  if writeFn[length(writeFn)] <> '\' then writeFn := writeFn + '\';
  //append filename to the target folder ...
  writeFn := writeFn + extractfilename(readFn);


  GetMem(buffer,SplitSize);
  if buffer = nil then
  begin
    SetStatusbarText(loadStr(sOutOfMemory));
    beep;
    exit; //something wrong!!
  end;

  try
    SetStatusbarText(loadStr(sSplittingFiles));
    //Open source file for reading...
    fReadHdl := FileOpen(readFn, fmOpenRead or fmShareDenyNone);
    if fReadHdl < 1 then
    begin
      SetStatusbarText(loadStr(sCantOpenFile));
      beep;
      exit; //something wrong!!
    end;

    oldCursor := SetCursor(LoadCursor(0, IDC_WAIT));
    try

      //get the checksum and write it to file...
      SetStatusbarText(loadStr(sCreatingChecksum));
      Checksum := GetChecksum(fReadHdl);
      //cancel if escape pressed while getting Checksum...
      if Checksum = $FFFFFFFF then exit;
      fWriteHdl := FileCreate(writeFn+'.999');
      if fWriteHdl > 0 then
      try
        FileWrite(fWriteHdl, Checksum, sizeof(Checksum));
      finally
        FileClose(fWriteHdl);
      end;

      //get the original file's datetime...
      GetFileTime(fReadHdl,nil,nil,@fModified);

      cnt := 1;
      //top of loop /////////////////////////////////
      repeat
        SetStatusbarText(loadStr(sSplittingFiles)+inttostr(cnt));
        ReadSize := FileRead(fReadHdl, Buffer^, SplitSize);
        splitFn := writeFn + format('.%3.3d',[cnt]);
        fWriteHdl := FileCreate(splitFn);
        if fWriteHdl < 1 then
        begin
          SetStatusbarText(loadStr(sCantCreateFile));
          beep;
          exit; //something wrong!!
        end;
        try
          FileWrite(fWriteHdl, Buffer^, ReadSize);
          //set to the original file's datetime...
          SetFileTime(fWriteHdl,nil,nil,@fModified);
        finally
          FileClose(fWriteHdl);
        end;
        inc(cnt);

        if EscapePressed then exit;
      until (ReadSize <> SplitSize);
      //bottom of loop //////////////////////////////
    finally
      FileClose(fReadHdl);
      SetCursor(oldCursor);
    end;
  finally
    FreeMem(Buffer);
  end;
  //copy self if checkbox ticked...
  if sendDlgItemMessage(SplitDlg, idMakeBatch, BM_GETCHECK,0,0) <> 0 then
  begin
    writeFn := extractfilepath(writeFn)+ extractfilename(paramstr(0));
    copyfile(pchar(paramstr(0)),pchar(writeFn),true); //ignored if already exists
  end;
  SetStatusbarText(loadStr(sSplitFinished));
end;
//--------------------------------------------------------------------------

procedure JoinFiles;
var
  cnt, FileSize, ReadSize, cntTotal: integer;
  OldChecksum, NewChecksum: dword;
  readFn, writeFn: string;
  fReadHdl, fWriteHdl: THandle;
  fModified: TFileTime;
  buffer: pchar;
  oldCursor: HCursor;
  joinSize: cardinal;
begin
  //disable 'OK' button...
  EnableWindow(GetDlgItem(joinDlg,idOKBtn), false);

  //get the filename for the new file...
  writeFn  := GetDlgItemCaption(JoinDlg, idSaveToEdit);
  //create the new file ready for writing...
  fWriteHdl := FileCreate(writeFn);
  if fWriteHdl < 1 then
  begin
    SetStatusbarText(loadStr(sCantCreateFile));
    beep;
    exit; //something wrong!!
  end;

  buffer := nil;
  oldCursor := SetCursor(LoadCursor(0, IDC_WAIT));
  try
    cntTotal := sendDlgItemMessage(JoinDlg, idListbox, LB_GETCOUNT,0,0);
    cnt := 0;
    joinSize := 0;
    SetStatusbarText(loadStr(sJoiningFiles));
    //top of loop /////////////////////////////////
    while (cnt < cntTotal) do
    begin
      SetStatusbarText(loadStr(sJoiningFiles)+inttostr(cnt+1));
      //get each filename part from the list...
      setlength(readFn,MAX_PATH);
      setlength(readFn, sendDlgItemMessage(JoinDlg,
        idListbox, LB_GETTEXT, cnt, longint(pchar(readFn))));
      if readFn = '' then break;
      readFn := joinSrcPath + readFn;
      fReadHdl := FileOpen(readFn,fmOpenRead or fmShareDenyNone);
      if fReadHdl < 1 then
      begin
        SetStatusbarText(loadStr(sCantOpenFile));
        beep;
        exit; //something wrong!!
      end;
      //now, copy this file...
      try
        //get the first file part's datetime...
        if cnt = 0 then GetFileTime(fReadHdl,nil,nil,@fModified);

        FileSize := Windows.GetFileSize(fReadHdl, nil);
        inc(joinSize,FileSize);
        ReAllocMem(buffer,FileSize);
        if buffer = nil then break;
        ReadSize := FileRead(fReadHdl, Buffer^, FileSize);
        if ReadSize <> FileSize then break;
        if FileWrite(fWriteHdl, Buffer^, FileSize) <> FileSize then break;
        if EscapePressed then exit;
      finally
        FileClose(fReadHdl);
      end;
      inc(cnt);
    end;
    //bottom of loop //////////////////////////////

    //check for errors while reading/writing files (see 'breaks' above) ...
    if joinSize <> Windows.GetFileSize(fWriteHdl, nil) then
    begin
      SetStatusbarText(loadStr(sCantReadWriteFile));
      beep;
      exit; //bad copy!!!
    end;

    //verify checksum if present...
    //(it wont be present if joining files split by another utility)
    readFn := changefileext(readFn,'.999');
    if fileexists(readFn) then
    begin
      SetStatusbarText(loadStr(sCheckingChecksum));
      NewChecksum := GetChecksum(fWriteHdl);

      if NewChecksum = $FFFFFFFF then //user cancelled checksum calc
        SetStatusbarText(loadStr(sJoinFinished))
      else
      begin
        fReadHdl := FileOpen(readFn,fmOpenRead or fmShareDenyNone);
        if fReadHdl > 0 then
        try
          if FileRead(fReadHdl, OldChecksum, sizeof(OldChecksum)) =
            sizeof(OldChecksum) then
              if NewChecksum = OldChecksum then
                SetStatusbarText(loadStr(sChecksumOK)) //OK!
              else
              begin
                SetStatusbarText(loadStr(sChecksumFail));
                beep;
                exit; //bad checksum!!!
              end;
        finally
          FileClose(fReadHdl);
        end;
      end;
    end
    else
      //if no checksum file, just notify join finished...
      SetStatusbarText(loadStr(sJoinFinished));

    //restore the file's original DateTime...
    SetFileTime(fWriteHdl,nil,nil,@fModified);
  finally
    FileClose(fWriteHdl);
    FreeMem(buffer);
    SetCursor(oldCursor);
  end;

  //delete all file parts if checkbox ticked...
  if sendDlgItemMessage(JoinDlg, idMakeBatch, BM_GETCHECK,0,0) <> 0 then
  begin
    //nb: readFn still = Checksum file at this point...
    if fileexists(readFn) then DeleteFile(readFn);
    cnt := 0;
    //top of loop /////////////////////////////////
    while cnt < cntTotal do
    begin
      //get each filename in the list...
      setlength(readFn,MAX_PATH);
      setlength(readFn, sendDlgItemMessage(JoinDlg,
        idListbox, LB_GETTEXT, cnt, longint(pchar(readFn))));
      if readFn = '' then break;
      DeleteFile(joinSrcPath + readFn);
      inc(cnt);
    end;
    //bottom of loop //////////////////////////////
  end;
end;
//--------------------------------------------------------------------------

function GetFileSize(const Filename: string): integer;
var
  ffHdl: THandle;
  wfd: TWIN32FINDDATA;
begin
  result := 0;
  if not fileexists(Filename) then exit;
  ffHdl := findfirstfile(pchar(Filename),wfd);
  if ffHdl = INVALID_HANDLE_VALUE then exit;
  result := wfd.nFileSizeLow;
  windows.FindClose(ffHdl);
end;
//--------------------------------------------------------------------------

//show and return the number of file parts ...
function UpdateSplitCount: integer;
var
  str: string;
  fs,ss: integer;
begin
  //get the name of the file to split...
  str  := GetDlgItemCaption(SplitDlg, idFileEdit);
  fs := GetFileSize(str);
  if (fs > 0) then
    ss := GetSplitSize else
    ss := 0;
  if (ss = 0) then
    result := 0 else
    result := (fs div (ss*1024)) +1;
  //also show the number of file parts...
  str := inttostr(result);
  sendDlgItemMessage(SplitDlg,idFileCountStatic,WM_SETTEXT,0,longint(pchar(str)));
end;
//--------------------------------------------------------------------------

function About(Dialog: HWnd; AMessage, WParam: LongInt; LParam: Longint): Bool; stdcall;
var
  str: string;
begin
  result := true;
  case AMessage of
    WM_INITDIALOG:
      begin
        //make 'SplitJoin' title bold...
        SendDlgItemMessage(Dialog, idBoldStatic, WM_SETFONT, boldFont, 0);
        //change the font & cursor for idWebStatic ...
        SendDlgItemMessage(Dialog, idwebStatic, WM_SETFONT, webFont, 0);
        SetClassLong(GetDlgItem(Dialog, idWebStatic),
          GCL_HCURSOR, LoadCursor( hInstance, MAKEINTRESOURCE(108)));
      end;
    //change idWebStatic's font color to blue...
    WM_CTLCOLORSTATIC:
      if lParam = longint(GetDlgItem(Dialog, idWebStatic)) then
      begin
        SetTextColor(wParam,RGB(0,0,255)); //color = blue   
        //now must also set the text background color & the static's color too ...
        SetBkColor(wParam,GetSysColor(COLOR_BTNFACE));
        result := bool(GetSysColorBrush(COLOR_BTNFACE));
      end;
    WM_COMMAND:
      case LoWord(WParam) of
        IDOK, IDCANCEL: EndDialog(Dialog, 1); //modal dialogs only

        //when my web address is clicked, open it using the default browser ...
        idWebStatic:
          if HiWord(WParam) = STN_CLICKED then
          begin
            str := GetDlgItemCaption(Dialog, idWebStatic);
            ShellExecute(0, nil, pchar(str), Nil, Nil, SW_NORMAL);
          end;
      end;
    else result := False; //not handled
  end;
end;
//--------------------------------------------------------------------------

type
  //used to stream in RTF text into the HelpContents dialog.
  PCookie = ^TCookie;
  TCookie = record
    Chars: PChar;
    Pos  : integer;
    Len  : integer;
  end;

//see the EM_STREAMIN Richedit message in Win32.Hlp for more info...
function RichEdStreamIn(dwCookie: Longint;
  pbBuff: PByte; cb: Longint; var pcb: Longint): Longint; stdcall;
var
  Cookie: PCookie;
begin
  Cookie := PCookie(dwCookie);
  pcb := Cookie.Len - Cookie.Pos;
  if pcb > cb then pcb := cb;
  try
    with Cookie^ do
    begin
      Move(Chars[Pos], pbBuff^, pcb);
      inc(Pos,pcb);
    end;
    result := 0; //OK
  except
    result := 1; //error
  end;
end;
//--------------------------------------------------------------------------

var
  OldRichedWindProc: pointer;

//HelpContents Richedit control subclassed ...
function NewRichedWindProc(hdl: THandle; msg, wParam, lParam: integer): longint; stdcall;
begin
  case msg of
    WM_SETFOCUS: result := 0; //stops the caret being displayed
    WM_KILLFOCUS: result := 0;
    WM_GETDLGCODE: result := DLGC_WANTARROWS;
    WM_KEYDOWN: //scroll the richedit up or down depending on arrow keys...
      begin
        result := 0;
        if wParam = VK_UP then
          sendmessage(hdl,EM_SCROLL, SB_LINEUP, 0)
        else if wParam = VK_DOWN then
          sendmessage(hdl,EM_SCROLL, SB_LINEDOWN, 0)
        else
          result := CallWindowProc(OldRichedWindProc, hdl, msg, wParam, lParam);
      end;
  else
    result := CallWindowProc(OldRichedWindProc, hdl, msg, wParam, lParam);
  end;
end;
//--------------------------------------------------------------------------

function Help(Dialog: HWnd; AMessage, WParam: LongInt; LParam: Longint): Bool; stdcall;
var
  es: TEDITSTREAM;
  Cookie: TCookie;
  ResHdl: THandle;
  ResGlob: THandle;
  ResText: pChar;

  RichEdHdl: THandle;
begin
  result := true;
  case AMessage of
    WM_INITDIALOG:            
      begin                      
        //load the RTF text from the resource into the Richedit control...
        ResHdl := FindResource(hInstance,'HELPINFO','TEXT');
        ResGlob := LoadResource(hInstance,ResHdl);
        ResText := LockResource(ResGlob);
        Cookie.Chars := ResText;
        Cookie.Pos := 0;
        Cookie.Len := strlen(ResText);
        es.dwCookie := LongInt(@Cookie);
        es.pfnCallBack := @RichEdStreamIn;
        es.dwError := 0;
        RichEdHdl := GetDlgItem(Dialog, idRichedit);
        SendMessage(RichEdHdl, EM_STREAMIN, SF_RTF, longint(@es));
        //change the Richedit background color ...
        SendMessage(RichEdHdl, EM_SETBKGNDCOLOR, 0, $d8ffff);
        //subclass the richedit window (hides the caret etc)...
        OldRichedWindProc := pointer(SetWindowLong(RichEdHdl,
          GWL_WNDPROC, longint(@NewRichedWindProc)));
      end;
    WM_COMMAND:
      case LoWord(WParam) of
        IDOK, IDCANCEL: EndDialog(Dialog, 1); //modal dialogs only
      end;
    else result := False; //not handled
  end;
end;
//--------------------------------------------------------------------------

function FileHasTrailingNumber(const Filename: string): boolean;
begin
  result := (Filename[length(Filename)] in ['0'..'9']);
end;
//--------------------------------------------------------------------------

procedure ProcessParameters(const Param1: string; Multifiles: boolean);
begin
  if Multifiles or FileHasTrailingNumber(Param1) then
  begin
    ShowTheTab(tsJoin);
    FillJoinListbox(Param1);
  end else
  begin
    ShowTheTab(tsSplit);
    sendDlgItemMessage(SplitDlg,
      idFileEdit, WM_SETTEXT, 0, longint(pchar(Param1)));
    sendDlgItemMessage(SplitDlg, idFileEdit, EM_SETSEL, 0, -1);
    SetFocus(GetDlgItem(SplitDlg,idFileEdit));
  end;
end;
//--------------------------------------------------------------------------

function Main(Dialog: HWnd; AMessage, WParam: LongInt; LParam: Longint): Bool; stdcall;
var
  buffer : array[0..MAX_PATH] of char;
  MultiFiles: boolean;
begin
  result := true;
  case AMessage of
    WM_INITDIALOG:
      begin
        //set the application icon...
        SetClassLong( Dialog, GCL_HICON, LoadIcon( hInstance, 'MAINICON'));
      end;
    WM_NOTIFY:
      with PNMHdr(lParam)^ do
        case idFrom of
          idTab: if (code = TCN_SELCHANGE) then
            //typecast the tabControls tab index to TTabShowing...
            ShowTheTab(TTabShowing(sendmessage(tab,TCM_GETCURSEL,0,0)));
        end;
    WM_COMMAND:
      case LoWord(WParam) of
        IDCANCEL: PostQuitMessage(1); //modeless dialogs only
        menuHelp: DialogBox(hInstance, MAKEINTRESOURCE(8), dlg , @Help);
        menuAbout: DialogBox(hInstance, MAKEINTRESOURCE(7), dlg , @About);
        accelSkip: ShowTheTab(tsSplit);
        accelJoin: ShowTheTab(tsJoin);
      end;
    //Respond to file drag and drop...
    WM_DROPFILES:
      begin
        MultiFiles := DragQueryFile(THandle(wParam), $FFFFFFFF, Nil, 0) > 1;
        DragQueryFile(THandle(wParam), 0, @buffer, sizeof(buffer));
        ProcessParameters(buffer,MultiFiles);
        SetForegroundWindow(Dialog);
      end;
    else result := False; //not handled
  end;
end;
//---------------------------------------------------------------------

function Split(Dialog: HWnd; AMessage, WParam: LongInt; LParam: Longint): Bool; stdcall;
var
  cnt: integer;
  str, fn: string;
  IsEqual: boolean;
begin
  result := true;
  case AMessage of
    WM_INITDIALOG: ;  //ie. Result = true. Otherwise The Browse button
                      //wont respond to the <enter> key initially.
    WM_COMMAND:
      case LoWord(WParam) of
        idFloppy:
          if (HIWORD(wParam) = BN_CLICKED) and
            (sendDlgItemMessage(SplitDlg, idFloppy, BM_GETCHECK,0,0) <> 0) then
            sendDlgItemMessage(SplitDlg, idSizeEdit,
              WM_SETTEXT, 0, longint(pchar(strFloppySize)));
        idFileEdit:
          if (HIWORD(wParam) = EN_CHANGE) then
          begin
            SetStatusbarText('');
            cnt := UpdateSplitCount;
            if IsOKToSplit and (cnt > 1) then
              EnableWindow(GetDlgItem(Dialog,idOKBtn), true) else
              EnableWindow(GetDlgItem(Dialog,idOKBtn), false);
          end;
        idSaveToEdit:
          if (HIWORD(wParam) = EN_CHANGE) then
          begin
            cnt := UpdateSplitCount;
            if IsOKToSplit and (cnt > 1) then
              EnableWindow(GetDlgItem(Dialog,idOKBtn), true) else
              EnableWindow(GetDlgItem(Dialog,idOKBtn), false);
          end;
        idFileBrowse:
          if (HIWORD(wParam) = BN_CLICKED) and
            OpenFileDialog(dlg,loadstr(sSelectSplitFile),
                loadStr(sAllFiles)+' (*.*)|*.*','',fn, false) then
          begin
            sendDlgItemMessage(SplitDlg,
                idFileEdit, WM_SETTEXT, 0, longint(pchar(fn)));
            sendDlgItemMessage(SplitDlg, idFileEdit, EM_SETSEL, 0, -1);
            fn := extractfilepath(fn);
            sendDlgItemMessage(SplitDlg,
                idSaveToEdit, WM_SETTEXT, 0, longint(pchar(fn)));
            SetFocus(GetDlgItem(SplitDlg,idFileEdit));
          end;
        idSaveToBrowse:
          if (HIWORD(wParam) = BN_CLICKED) then
          begin
            fn := GetDlgItemCaption(SplitDlg, idSaveToEdit);
            if not GetFolder(Dialog, fn) then exit;
            sendDlgItemMessage(SplitDlg, idSaveToEdit,
              WM_SETTEXT, 0, longint(pchar(fn)));
            sendDlgItemMessage(SplitDlg, idSaveToEdit, EM_SETSEL, 0, -1);
            SetFocus(GetDlgItem(SplitDlg,idSaveToEdit));
          end;
        idSizeEdit:
          if (HIWORD(wParam) = EN_CHANGE) then
          begin
            cnt := UpdateSplitCount;
            if IsOKToSplit and (cnt > 1) then
              EnableWindow(GetDlgItem(Dialog,idOKBtn), true) else
              EnableWindow(GetDlgItem(Dialog,idOKBtn), false);
            //uncheck the Floppy checkbox if <> strFloppySize ...
            str := GetDlgItemCaption(SplitDlg, idSizeEdit);
            IsEqual := strcomp(pchar(strFloppySize),pchar(str)) = 0;
            sendDlgItemMessage(SplitDlg,idFloppy,BM_SETCHECK,integer(IsEqual),0);
          end;
        idOKBtn: SplitFiles;
      end;
    else result := False; //not handled
  end;
end;
//---------------------------------------------------------------------

function Join(Dialog: HWnd; AMessage, WParam: LongInt; LParam: Longint): Bool; stdcall;
var
  i: integer;
  fn: string;
begin
  result := true;
  case AMessage of
    WM_INITDIALOG: ; //result := true;
    WM_COMMAND:
      case LoWord(WParam) of
        idFileBrowse:
          if (HIWORD(wParam) = BN_CLICKED) then
          begin
            SetStatusbarText('');
            if OpenFileDialog(dlg,loadstr(sSelectJoinFiles),
            loadStr(sAllFiles)+' (*.*)|*.*','',fn,true)
            and FillJoinListbox(fn) then
              EnableWindow(GetDlgItem(Dialog,idOKBtn), IsOKToJoin);
          end;
        idSaveToEdit:
          if (HIWORD(wParam) = EN_CHANGE) then
            EnableWindow(GetDlgItem(Dialog,idOKBtn), IsOKToJoin);
        idSaveToBrowse:
          if (HIWORD(wParam) = BN_CLICKED) then
          begin
            //suggest a SAVEAS filename...
            if sendDlgItemMessage(JoinDlg, idListbox, LB_GETCOUNT,0,0) > 0 then
            begin
              setlength(fn,MAX_PATH);
              setlength(fn, sendDlgItemMessage(JoinDlg,
                idListbox, LB_GETTEXT, 0, longint(pchar(fn))));
              i := length(fn);
              while (i > 0) and (fn[i] in ['0'..'9']) do dec(i);
              if (i > 0) and (fn[i] = '.') then dec(i);
              fn := copy(fn,1,i);
            end;
            if SaveFileDialog(dlg,loadstr(sNewJoinFilename),
              '','',fn) then
            begin
              sendDlgItemMessage(JoinDlg,
                idSaveToEdit, WM_SETTEXT, 0, longint(pchar(fn)));
              sendDlgItemMessage(Dialog, idSaveToEdit, EM_SETSEL, 0, -1);
              SetFocus(GetDlgItem(Dialog,idSaveToEdit));
            end;
          end;
        idOKBtn: JoinFiles;
      end;
    else result := False; //not handled
  end;
end;
//---------------------------------------------------------------------

//--------------------------------------------------------------------------
// Main prog. entry point...
//--------------------------------------------------------------------------
var
  msg: TMsg;
  tcItem: TTcItem;
  childDlgResult: boolean;
  path: string;
  ffHdl: THandle;
  wfd: TWIN32FINDDATA;
  RichEdLib: THandle;
begin
  osVersionInfo.dwOSVersionInfoSize := sizeof(osVersionInfo);
  GetVersionEx(osVersionInfo);

  initCommonControls; //needed because of "SysTabControl32" on main dialog

  RichEdLib := LoadLibrary('Riched32.dll'); //needed for the Help dialog

  //create the main dialog (using Dialog resource #6)...
  dlg := CreateDialog( hInstance, MAKEINTRESOURCE(6), 0, @Main);
  if dlg = 0 then exit;

  accel := LoadAccelerators(hInstance,MAKEINTRESOURCE(4096));

  //Create a bold font for the about dialog...
  boldFont := CreateFont(-28, 0, 0, 0, FW_EXTRABOLD, 0, 0, 0, DEFAULT_CHARSET,
                      OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY,
                      DEFAULT_PITCH or FF_DONTCARE, pchar(loadStr(sFontface)));

  //Create a web font (bold & underlined) for the about dialog...
  //( nb: coloring text blue is done in About(). )
  webFont := CreateFont(-11, 0, 0, 0, FW_BOLD, 0, 1, 0, DEFAULT_CHARSET,
                      OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY,
                      DEFAULT_PITCH or FF_DONTCARE, pchar(loadStr(sFontface)));

  //add the tabs to the "SysTabControl32" control...
  tab := GetDlgItem(dlg,idtab);
  tcItem.mask := TCIF_TEXT;
  tcItem.pszText := pchar(Loadstr(sSplit));
  sendmessage(tab,TCM_INSERTITEM,0,longint(@tcItem));
  tcItem.pszText := pchar(Loadstr(sJoin));
  sendmessage(tab,TCM_INSERTITEM,1,longint(@tcItem));

  //create 2 child dialogs (with one hidden for the time being)...
  SplitDlg := CreateDialog( hInstance, MAKEINTRESOURCE(12), tab, @Split);
  JoinDlg := CreateDialog( hInstance, MAKEINTRESOURCE(14), tab, @Join);

  //TabShowing = tsJoin ensures that the following
  //ShowTheTab() doesn't exit immediately...
  TabShowing := tsJoin;
  ShowTheTab(tsSplit);

  //set the initial split size to the size of a floppy disk...
  sendDlgItemMessage(SplitDlg, idSizeEdit,
    WM_SETTEXT, 0, longint(pchar(strFloppySize)));


  if (paramcount > 0) then
    ProcessParameters(GetLongFilename(paramstr(1)),paramcount > 1)
  else
  begin
    //check if there are split files in the current directory...
    path := extractfilepath(paramstr(0));
    ffHdl := findfirstfile(pchar(path +'*.001'),wfd);
    if ffHdl <> INVALID_HANDLE_VALUE then
    try
      if (wfd.dwFileAttributes and FILE_ATTRIBUTE_DIRECTORY) = 0 then
        ProcessParameters(path + wfd.cFileName, true);
    finally
      windows.FindClose(ffHdl);
    end;
  end;

  //enable drag and drop... ( see also WM_DROPFILES in Main(). )
  DragAcceptFiles(Dlg, True);

  //now, to handle 3 concurrent modeless DialogProcs
  //in the main message loop ...
  while GetMessage(msg, 0, 0, 0) do
  begin
    if TranslateAccelerator(dlg,accel,msg) <> 0 then continue;
    //the main dialog owns 2 child dialogs
    //(however, only one child is visible at a time)...
    if TabShowing = tsSplit then
      childDlgResult := IsDialogMessage(SplitDlg, msg)
    else
      childDlgResult := IsDialogMessage(JoinDlg, msg);
    if not childDlgResult then IsDialogMessage(dlg, msg);
  end;

  //cleanup...
  FreeLibrary(RichEdLib);
  DeleteObject(boldFont);
  DeleteObject(webFont);
  DestroyWindow(splitDlg);
  DestroyWindow(joinDlg);
  DestroyWindow(dlg);
end.

