#!/usr/bin/perl -w
#
# Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
#
# startallvms.pl
#
# This script powers on all VMs on the system that are not
# already running.
#
# usage: 
#   startallvms.pl <hostname> <user> <password>
#

BEGIN {
   if ($^O eq "MSWin32") {
      @INC = (
         # Set the path to your VmPerl Scripting directory if different
         'C:\Program Files\VMware\VMware VmPerl Scripting API\perl5\site_perl\5.005',
         'C:\Program Files\VMware\VMware VmPerl Scripting API\perl5\site_perl\5.005\MSWin32-x86');
   }
}

use VMware::VmPerl;
use VMware::VmPerl::VM;
use VMware::VmPerl::Server;
use VMware::VmPerl::ConnectParams;
use strict;

my ($server_name, $user, $passwd) = @ARGV;

# Change this to your port if it is different.
my $port = 902;

# Create a ConnectParams object
my $connect_params =
   VMware::VmPerl::ConnectParams::new($server_name,$port,$user,$passwd);

# Create a Server object
my $server = VMware::VmPerl::Server::new();

# Establish a persistent connection with server
if (!$server->connect($connect_params)) {
   my ($error_number, $error_string) = $server->get_last_error();
   die "Could not connect to server: Error $error_number: $error_string\n";
}

# Get a list of all virtual machine configuration files registered
# with the server.
my @list = $server->registered_vm_names();


if(!defined($list[0])) {
   my ($error_number, $error_string) = $server->get_last_error();
   die "Could not get list of VMs: Error $error_number: $error_string\n";
}

my $config;

foreach $config (@list) {

   my $vm = VMware::VmPerl::VM::new();

   # Connect to the VM, using the same ConnectParams object.
   if (!$vm->connect($connect_params, $config)) {
      my ($error_number, $error_string) = $server->get_last_error();
      print STDERR "Could not connect to VM $config: Error $error_number: ".
                   "$error_string\n";
   } else {
      # Only power on VMs with the config setting autostart = "true"
      my $autostart = $vm->get_config("autostart");

      if($autostart && ($autostart =~ /true/i || $autostart =~ /poweron/i)) {

	 # Only try this for VMs that are powered off or suspended.
	 my $power_state = $vm->get_execution_state();

	 if (!defined($power_state)) {
	    my ($error_number, $error_string) = $server->get_last_error();
	    print STDERR "Could not get execution state of VM $config: Error ".
			 "$error_number: $error_string\n";
	 } elsif ($power_state == VM_EXECUTION_STATE_OFF ||
	     $power_state == VM_EXECUTION_STATE_SUSPENDED) {

	    print "Powering on $config...\n";
	    if (!$vm->start()) {
	       # If an error occurs, report it and continue
	       my ($error_number, $error_string) = $server->get_last_error();
	       print STDERR "Could not power on VM $config: Error ".
                            "$error_number: $error_string\n";
	    } else {

	       # Delay slightly between starting each VM.
               # This prevents too much initial load on the server.
               
               # Warning: starting many VMs in rapid succession
               # is not recommended.

	       sleep 5;
            }
	 }
      }

      # Destroys the virtual machine object, thus disconnecting from the virtual machine.
      undef $vm;
   }
}

# Destroys the server object, thus disconnecting from the server.
undef $server;
