# GNU Enterprise Forms - Curses UI Driver - Button Widget
#
# Copyright 2000-2009 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 3, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: button.py 9956 2009-10-11 18:54:57Z reinhard $

import curses

from _base import UIHelper

__all__ = ['UIButton']

# =============================================================================
# Button class
# =============================================================================

class UIButton(UIHelper):

    # -------------------------------------------------------------------------
    # Initialization
    # -------------------------------------------------------------------------

    def __init__(self, event):

        UIHelper.__init__(self, event)

        self.__enabled = {}

    # -------------------------------------------------------------------------
    # Initialization per row
    # -------------------------------------------------------------------------

    def _create_widget_(self, event, spacer):

        UIHelper._create_widget_(self, event, spacer)
        self.__enabled[spacer] = True

    # -------------------------------------------------------------------------
    # Enable/disable this button
    # -------------------------------------------------------------------------

    def _ui_enable_(self, index):
        self.__enabled[index] = True
        self.__repaint(index, False)

    # -------------------------------------------------------------------------

    def _ui_disable_(self, index):
        self.__enabled[index] = False
        self.__repaint(index, False)

    # -------------------------------------------------------------------------
    # Focus has changed to this button
    # -------------------------------------------------------------------------

    def _ui_focus_in_(self, index):

        curses.curs_set(0)
        self.__repaint(index, True)

    # -------------------------------------------------------------------------
    # Focus has changed away from this button
    # -------------------------------------------------------------------------

    def _ui_focus_out_(self, index):

        curses.curs_set(1)
        self.__repaint(index, False)

    # -------------------------------------------------------------------------
    # Update button representation on screen
    # -------------------------------------------------------------------------

    def __repaint(self, index, focused):

        if not self.ready():
            return

        if focused:
            attr = self._uiDriver.attr['focusentry']
        elif not self.__enabled[index]:
            attr = self._uiDriver.attr['disabled']
        else:
            attr = self._uiDriver.attr['entry']

        if focused:
            self._parent.move(self.left, self.top + index)
        self._set_text(index, self.__text, attr)

    # -------------------------------------------------------------------------

    def _keypress(self, key):

        if key in [' ', '\n']:
            self._gfObject._event_fire()
        else:
            UIHelper._keypress(self, key)

    def get_size_hints(self, vertical=None):

        return (self.min_width or 20, 1, 0, 0)

    def set_size_and_fit(self, width, height):

        self.width = width
        self.height = height
        # Determine button text
        maxlen = width - 2
        label = self._gfObject.label[:maxlen]              # cut if too long
        label = ' ' * ((maxlen - len(label)) / 2) + label  # expand if too short
        label = label + ' ' * (maxlen - len(label))

        self.__text = '[' + label + ']'

        for index in range(len(self.widgets)):
            self.__repaint(index, False)


# =============================================================================
# Configuration data
# =============================================================================

configuration = {
  'baseClass': UIButton,
  'provides' : 'GFButton',
  'container': 0,
}
