# GNU Enterprise Forms - QT3 UI Driver - GridLine widget
#
# Copyright 2001-2009 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 3, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: gridline.py 9956 2009-10-11 18:54:57Z reinhard $
"""
Implementation of a <gridline>.  A gridline is a logical line within a <grid>
and defines the controls building up the line.
"""

import qt

from gnue.forms.uidrivers.qt3.widgets import _base

__all__ = ['UIGridLine']

# =============================================================================
# Interface implementation for a grid line widget
# =============================================================================

class UIGridLine (_base.UIHelper):
    """
    Collection of controls building a given line in a grid.

    @ivar _columns: a dictionary which holds a sequence of QFrame instances
        per spacer.  Each of these panels will get the parent widget for the
        control located in that cell.
    """

    # -------------------------------------------------------------------------
    # Constructor
    # -------------------------------------------------------------------------

    def __init__(self, event):

        _base.UIHelper.__init__(self, event)
        self._columns = {}


    # -------------------------------------------------------------------------
    # Create the line widget
    # -------------------------------------------------------------------------

    def _create_widget_ (self, event, spacer):
        """
        Prepare the columns (cells) for the girdline and add it to the owner.
        """

        parent = event.container
        self.__prepare_columns(parent, spacer)
        self._container = parent

        return None


    # -------------------------------------------------------------------------
    # Prepare the cells for this gridline
    # -------------------------------------------------------------------------

    def __prepare_columns(self, parent, spacer):

        owner = self.getParent()
        sizer = parent.layout()
        row = owner._get_row(self, spacer)

        offset = 0
        for child in self._children:
            panel = qt.QWidget(parent)
            hbx = qt.QHBoxLayout(panel)

            if not spacer % 2:
                name = 'grid_color_even'
            else:
                name = 'grid_color_odd'

            color = self.__load_color_from_string(gConfigForms(name))
            panel.setPaletteBackgroundColor(color)

            self._columns.setdefault(spacer, []).append(panel)

            sizer.addMultiCellWidget(panel, row, row, offset, offset +
                    child.span)
            offset += child.span


    # -------------------------------------------------------------------------
    # Load the given colorname form the database or parse it as hex-rgb-string
    # -------------------------------------------------------------------------

    def __load_color_from_string(self, value):

        result = qt.QColor(value)
        if not result.isValid():
            (red, green, blue) = value[:2], value[2:4], value[4:6]
            result = qt.QColor(int(red, 16), int(green, 16), int(blue, 16))

        return result


    # -------------------------------------------------------------------------
    # Add an UI widget to the Grid container
    # -------------------------------------------------------------------------

    def add_widgets(self, ui_widget, spacer):
        """
        Add a given UI widget to the gridline.

        @param ui_widget: widget to add to the page
        @param spacer: the row-index (relative record number) to add the widget 
        """

        panel = self._get_cell(ui_widget, spacer)
        sizer = panel.layout()

        if ui_widget.label:
            sizer.addWidget(ui_widget.label)
        sizer.addWidget(ui_widget.widget)

        if ui_widget.label:
            ui_widget.widget.hide()


    # -------------------------------------------------------------------------
    # Get the cell-parent for a given child-control
    # -------------------------------------------------------------------------

    def _get_cell(self, ui_widget, spacer):
        """
        Return the wx.Panel instance (acting as parent widget) for a given
        UI-Widget.

        @param ui_widget: the UIWidget to get the cell parent for
        @param spacer: the spacer of the row to get the cell parent for
        """
        index = self._children.index(ui_widget)
        return self._columns[spacer][index]


# =============================================================================
# Configuration data
# =============================================================================

configuration = {
  'baseClass': UIGridLine,
  'provides' : 'GFGridLine',
  'container': 1
}
