# GNU Enterprise Reports - Run/create a user interface for starting a report
#
# Copyright 2001-2009 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 3, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: GRRunUI.py 10125 2009-12-11 17:58:25Z reinhard $

import datetime
import string
import StringIO

from gnue.common.apps import errors
from gnue.common.apps.i18n import utranslate as u_
from gnue.common.utils.FileUtils import dyn_import
from gnue.common.datasources.GLoginHandler import BasicLoginHandler


# =============================================================================
# This class handles an UI for a report
# =============================================================================

class GRRunUI:

  _DTYPES = [('file'   , u_("File")),
             ('printer', u_("Printer")),
             ('email'  , u_("E-Mail"))]

  # --------------------------------------------------------------------------
  # Constructor
  # --------------------------------------------------------------------------

  def __init__ (self, reportInfo, cfgManager, connections, ui = 'text',
      userParameters = {}, destination = '-', destinationType = 'file',
      filter = 'raw', sortoption = None):

    self.configurationManager = cfgManager
    self.connections          = connections

    if ui != 'text':
      try:
        self.uimodule = dyn_import ('gnue.forms.uidrivers.%s' % ui)

        # Nasty hackery
        from gnue.forms.GFConfig import ConfigOptions as FormConfigOptions
        cfgManager.loadApplicationConfig (section = "forms",
                                                  defaults = FormConfigOptions)
        cfgManager.registerAlias ('gConfigForms', 'forms')

        self.gfinstance = dyn_import ('gnue.forms.GFInstance')

      except ImportError:
        ui = 'text'


    if ui == 'text':
      self.uimodule     = None

      from GRRun import quietprint, _quietprintSuppress
      silent = quietprint == _quietprintSuppress

      self.loginhandler = BasicLoginHandler (True, silent)
    else:
      self.loginhandler = self.uimodule.UILoginHandler ()

    self.reportInfo      = reportInfo
    self.userParameters  = userParameters
    self.ui              = ui
    self.destination     = destination
    self.destinationType = destinationType
    self.filter          = filter
    self.sortoption      = sortoption


  # ---------------------------------------------------------------------------
  # create a user interface and run it
  # ---------------------------------------------------------------------------

  def run (self):
    """
    """

    if self.uimodule:
      # First add some internal parameters to the userparams-dictionary
      self.userParameters ['filter']          = self.filter
      self.userParameters ['destination']     = self.destination
      self.userParameters ['destinationtype'] = self.destinationType
      self.userParameters ['ok']              = False

      # Create the form's code on the fly and put it into an instance
      formBuffer = self.__buildForm ()
      instance   = self.gfinstance.GFInstance (self, self.connections,
                                      self.uimodule, True)
      instance.run_from_buffer(formBuffer, self.userParameters)

      # Now iterate through the parameters dictionary and fetch our 'private'
      # values, also remove them from userParameters
      if not self.userParameters ['ok']:
        raise errors.UserError, u_("Report canceled")

      self.filter          = self.userParameters ['filter']
      self.destination     = self.userParameters ['destination']
      self.destinationType = self.userParameters ['destinationtype']

      del self.userParameters ['filter']
      del self.userParameters ['destination']
      del self.userParameters ['destinationtype']
      del self.userParameters ['ok']


  # ---------------------------------------------------------------------------
  # Get the current login handler
  # ---------------------------------------------------------------------------

  def getLoginHandler (self):
    """
    """

    return self.loginhandler


  # ---------------------------------------------------------------------------
  # Get the user parameter dictionary
  # ---------------------------------------------------------------------------

  def getParameters (self):
    """
    """

    return self.userParameters


  # ---------------------------------------------------------------------------
  # Get the current destination
  # ---------------------------------------------------------------------------

  def getDestination (self):
    """
    """

    return self.destination


  # ---------------------------------------------------------------------------
  # Get the destination type
  # ---------------------------------------------------------------------------

  def getDestinationType (self):
    """
    """

    return self.destinationType


  # ---------------------------------------------------------------------------
  # Get the current filter
  # ---------------------------------------------------------------------------

  def getFilter (self):
    """
    """

    return self.filter


  # ---------------------------------------------------------------------------
  # Get the current sort option
  # ---------------------------------------------------------------------------

  def getSortOption (self):
    """
    """

    return self.sortoption


  # ---------------------------------------------------------------------------
  # Generate a form for entering parameters and filter/destination
  # ---------------------------------------------------------------------------

  def __buildForm (self):
    """
    This function generates a GNU Enterprise Form Definition for the report as
    described by our reportInfo property.

    @return: filehandle to read the form from
    """

    code = ['<?xml version="1.0" encoding="utf-8"?>']
    code.extend (self._getXMLTag ('form', \
        {'title': self.reportInfo.getTitle (),
         'style': 'dialog'}, "", True))

    # Add general parameters
    code.extend(self._getXMLTag('parameter',
            {'name': 'filter', 'datatype': 'text'}))
    code.extend(self._getXMLTag('parameter',
            {'name': 'destination', 'datatype': 'text'}))
    code.extend(self._getXMLTag('parameter',
            {'name': 'destinationtype', 'datatype': 'text'}))
    code.extend(self._getXMLTag('parameter',
            {'name': 'ok', 'datatype': 'boolean'}))

    fields = self.__translateParameters ()

    # Add all parameters as 'user parameters' to the form definition
    for item in fields:
      code.extend (self._getXMLTag ('parameter', \
          {'name': item ['name'], 'default': item ['default']}, "  "))

    # Add fields for the filter
    self.__addOtherFields (fields)

    # Get some positioning information from all fields
    formHeight      = len (fields) + 3
    self.labelWidth = 0
    formWidth       = 0

    for item in fields:
      self.labelWidth = max (self.labelWidth, len (item ['label']))
      formWidth  = max (formWidth, self.labelWidth + 2 + item ['dLength'])

    # Add datasources and blocks
    code.extend (self.__addDataSources ())
    code.extend (self._getXMLTag ('logic', \
                                {'contents': self.__addBlocks (fields)}, "  "))

    code.extend (self.__addTriggers (fields))

    # Layout
    code.extend (self._getXMLTag ('layout', \
        {'xmlns:c' : 'GNUe:Layout:Char',
         'c:width' : formWidth,
         'c:height': formHeight,
         'contents': self.__addLabelsAndEntries ("Parameters", fields)},
         "  "))

    code.append ("</form>")

    # Prepare a buffer with the form's code and set it's position to the start
    stream = string.join (code, "\n")

    return stream.encode ('utf-8')


  # ---------------------------------------------------------------------------
  # Translate the report parameters dictionary into a sequence
  # ---------------------------------------------------------------------------

  def __translateParameters (self):
    """
    This function maps all report parameters to a sequence of field
    definitions.

    @return: sequence of dictionaries, one per parameter
    """

    result = []

    for param in self.reportInfo.getParameters ():
      name    = param ['name']
      
      current = {'name'    : name,
                 'default' : self.userParameters.get (name, param ['default']),
                 'style'   : None,
                 'format'  : None,
                 'typecast': None,
                 'dLength' : 35,
                 'pType'   : param.get ('type', 'char'),
                 'skip'    : False,
                 'fieldDef': []}

      pType = current ['pType']

      if pType == 'boolean':
        current ['style']   = 'checkbox'
        current ['dLength'] = 1

      elif pType in ['date', 'time', 'datetime']:
        current ['typecast'] = 'date'
        masks  = ['%x', '%X', '%x %X']
        format = masks [['date', 'time', 'datetime'].index (pType)]
        current ['format']  = format
        current ['dLength'] = len (datetime.datetime.now().strftime(format))

      current ['label'] = param.get ('description', name)
      result.append (current)

    return result


  # ---------------------------------------------------------------------------
  # Add all other form fields to the field-sequecnce
  # ---------------------------------------------------------------------------

  def __addOtherFields (self, fields):
    """
    This function extends the given field-sequence with all outstanding fields
    of the form, which are not given as paramter.

    @param fields: sequence of field dictionaries to be extended.
    """

    fields.append ({'name'    : 'filter',
                    'label'   : u_("Output filter"),
                    'style'   : 'dropdown',
                    'format'  : None,
                    'typecast': None,
                    'dLength' : 35,
                    'pType'   : 'char',
                    'skip'    : False,
                    'fieldDef': [{'fk_source': 'dtsFilters'},
                                 {'fk_key': 'key'},
                                 {'fk_description': 'descr'}]})

    fields.append ({'name'    : 'destinationtype',
                    'label'   : u_("Destination type"),
                    'style'   : 'dropdown',
                    'format'  : None,
                    'typecast': None,
                    'dLength' : 35,
                    'pType'   : 'char',
                    'skip'    : False,
                    'fieldDef': [{'fk_source': 'dtsDestinations'},
                                 {'fk_key': 'key'},
                                 {'fk_description': 'descr'}]})

    fields.append ({'name'    : 'destination',
                    'label'   : u_("Destination"),
                    'style'   : None,
                    'format'  : None,
                    'typecast': None,
                    'dLength' : 35,
                    'pType'   : 'char',
                    'skip'    : False,
                    'fieldDef': []})


  # ---------------------------------------------------------------------------
  # Create the XML code for all datasources needed by the form
  # ---------------------------------------------------------------------------

  def __addDataSources (self):
    """
    This function generates a XML code sequence with all datasources needed by
    the form.

    @return: sequence of XML code lines
    """

    filters = [('raw', u_("Raw XML"))]
    if self.reportInfo.getNamespace ():
      for (key, item) in self.reportInfo.getFilters ():
        name  = key.split (':') [-1]
        descr = item.get ('description', u_("%s output filter") % name)

        filters.append ((name, descr))

    rows = []
    for (key, descr) in filters:
      rows.extend (self.__getStaticSetRow (key, descr))

    result = self._getXMLTag ('datasource', {'name': 'dtsFilters',
        'type': 'static', 'prequery': 'Y',
        'contents': self._getXMLTag ('staticset', {'fields': 'key,descr',
          'contents': rows}, "")}, "  ")

    rows = []
    for (key, descr) in self._DTYPES:
      rows.extend (self.__getStaticSetRow (key, descr))

    result.extend (self._getXMLTag ('datasource', {'name': 'dtsDestinations',
      'type': 'static', 'prequery': 'Y',
      'contents': self._getXMLTag ('staticset', {'fields': 'key,descr',
        'contents': rows})}, "  "))


    return result


  # ---------------------------------------------------------------------------
  # Create a row in a static set for the given key and value
  # ---------------------------------------------------------------------------

  def __getStaticSetRow (self, key, val):
    """
    """

    k = self._getXMLTag ('staticsetfield', {'name': 'key', 'value': key}, "")
    d = self._getXMLTag ('staticsetfield', {'name': 'descr', 'value': val}, "")

    return self._getXMLTag ('staticsetrow', {'contents': k + d})


  # ---------------------------------------------------------------------------
  # Add the XML code for all blocks needed
  # ---------------------------------------------------------------------------

  def __addBlocks (self, fields):
    """
    This function generates all block definitions.

    @return: XML code sequence with block definitions
    """

    # First build the block with all input fields
    inpBlock = []

    for item in fields:
      attrs = {'name': item ['name']}
      for additional in item ['fieldDef']:
        attrs.update (additional)
      if item ['typecast'] is not None:
        attrs ['typecast'] = item ['typecast']

      inpBlock.extend (self._getXMLTag ('field', attrs, ""))

    result = self._getXMLTag ('block', {'name': 'blkInput', 'autoCommit': 'Y',
                                        'contents': inpBlock})

    return result

    
  # ---------------------------------------------------------------------------
  # Add the XML code for all triggers
  # ---------------------------------------------------------------------------

  def __addTriggers (self, fields):
    """
    This function creates the global triggers.

    @return: XML code sequence with trigger code
    """

    onActivate = []
    onExit     = []

    for item in fields:
      if item ['skip']:
        continue

      onActivate.append ("blkInput.%s.set (getParameter ('%s'))" \
                          % (item ['name'], item ['name']))

      onExit.append ("setParameter ('%s', blkInput.%s.get ())" \
                     % (item ['name'], item ['name']))

    
    result = []

    onStartup = ["form.setFeature ('GUI:MENUBAR:SUPPRESS', 1)",
                 "form.setFeature ('GUI:TOOLBAR:SUPPRESS', 1)",
                 "form.setFeature ('GUI:STATUSBAR:SUPPRESS', 1)"]

    result.extend (self._getXMLTag ('trigger', {'type': 'ON-STARTUP',
                                                'contents': onStartup}, "  "))
    if len (onActivate):
      result.extend (self._getXMLTag ('trigger', {'type': 'ON-ACTIVATION',
                                                'contents': onActivate}, "  "))
    if len (onExit):
      result.extend (self._getXMLTag ('trigger', {'type': 'ON-EXIT',
                                                'contents': onExit}, "  "))

    # Button-Triggers
    result.extend (self._getXMLTag ('trigger', {'type': 'NAMED',
      'name': 'trgOk',
      'contents': ["setParameter ('ok', True)", "close ()"]}, "  "))

    result.extend (self._getXMLTag ('trigger', {'type': 'NAMED',
      'name': 'trgCancel', 'contents': ["close ()"]}, "  "))

    return result


  # ---------------------------------------------------------------------------
  # Add all entries and labels to the given page
  # ---------------------------------------------------------------------------

  def __addLabelsAndEntries (self, page, fields):
    """
    This function creates a page with all the fields listed.

    @param page: Name of the page to add the fields to
    @param fields: sequence of field dictionaries to be added to the page

    @return: XML code sequence with all fields including an Ok and Cancel
        button.
    """

    result = []

    labels  = []
    entries = []
    row     = 0

    for item in fields:
      row  += 1

      # Label
      attrs = {'c:x'     : 1,
               'c:y'     : row,
               'c:width' : len (item ['label']),
               'c:height': 1,
               'text'    : "%s:" % item ['label']}
      labels.extend (self._getXMLTag ('label', attrs))

      # Entry
      attrs = {'c:x'     : 1 + self.labelWidth,
               'c:y'     : row,
               'c:width' : item ['dLength'],
               'c:height': 1,
               'block'   : 'blkInput',
               'field'   : item ['name']}

      if item ['style'] is not None:
        attrs ['style'] = item ['style']

      if item ['format'] is not None:
        attrs ['displaymask'] = item ['format']
        attrs ['inputmask']   = item ['format']

      entries.extend (self._getXMLTag ('entry', attrs))

    row += 2
    buttons = [('Ok', u_("Ok")), ('Cancel', u_("Cancel"))]
    maxbWidth = 0
    for b in buttons:
      maxbWidth = max (maxbWidth, len (b [1]) + 4)

    left = self.labelWidth + 1

    for b in buttons:
      attrs = {'c:x'     : left,
               'c:y'     : row,
               'c:width' : maxbWidth,
               'c:height': 1,
               'label'   : b [1],
               'contents': self._getXMLTag ('trigger', {'src': "trg%s" % b [0],
                                                        'type': 'ON-ACTION'})}

      entries.extend (self._getXMLTag ('button', attrs))
      left += (maxbWidth + 1)

    return self._getXMLTag ('page', \
                                  {'name': page, 'contents': labels + entries})

  # ---------------------------------------------------------------------------
  # Create an XML sequence
  # ---------------------------------------------------------------------------

  def _getXMLTag (self, tag, attrs, indent = "", keep = False):
    """
    This function creates a code-sequence for the given XML tag.

    @param tag: name of the XML tag
    @param attrs: dictionary with attributes
    @param indent: indentation for each line
    @param keep: if TRUE the tag is not closed, although attrs has no contents
        key.

    @return: sequence with XML code, one element per line
    """
    result = []
    parts  = []
    gap    = "  "

    if attrs.has_key ('contents'):
      contents = []
      for element in attrs ['contents']:
        contents.extend (element.splitlines ())
      del attrs ['contents']
    else:
      contents = None

    keys = attrs.keys ()
    keys.sort ()
    if 'name' in keys:
      keys.remove ('name')
      keys.insert (0, 'name')

    close = (contents is None and not keep) and "/" or ""

    xmlString = "%s<%s" % (indent, tag)
    nextIndent = len (xmlString)

    for key in keys:
      add = '%s="%s"' % (key, attrs [key])
      if len (xmlString) + len (add) > 76:
        result.append (xmlString)
        xmlString = " " * nextIndent

      xmlString = "%s %s" % (xmlString, add)

    xmlString = "%s%s>" % (xmlString, close)
    result.append (xmlString)

    if contents is not None:
      iLen = len (indent) + len (gap)
      cString = string.join (["%s%s" % (" " * iLen, i) for i in contents], "\n")

      result.append (cString)
      result.append ("%s</%s>" % (indent, tag))

    return result
